////////////////////////////////////////////////////////////
//
//    RUH KAPANI v0.1
//
//    Mehmet Deniz Aydinoglu . 2004
//    (deniz@oyunyapimi.org)
//
//    Lisans: GPL (license.txt)
//
//    www.oyunyapimi.org
//
////////////////////////////////////////////////////////////

#include "App.h"

static void message(const char *str);

//--------------------------------------------------------

// UYARI: golge olayi:
// golge icin kullanilan projeksiyon matrisini olusturan fonksiyon setShadowMatrix
// fonksiyonudur. Bu fonksiyon sadece birkez cagirilir.
// Projeksiyon matrisinden kasit GL_PROJECTION isleminde kullanilan tarzdaki
// bir 3d-2d cevirim matrisi degildir! Bu matris bildigimiz anlamda bir transformasyon
// matrisidir (GL_MODELVIEW e etki edecek!). Gorevi ise aktif hale getirildigi zaman
// uygulandigi noktalari belirtilen taban ustune gelecek sekilde konumlandirmak

// golgenin projekte edecegi yuzey dekleminin parametreleri. (normali {0,1,0} olan XZ duzlemi)
const float Plane[4] = {0,1,0,0};

// golge icin kullanilan isik pozisyonu.
const float LightPosition[] = {200.0f, 300.1f, 200.0f, 1.0f};

// golge icin kullanilan projeksiyon matrisi. Bu matris ile bir cismi
// golgenin dusecegi taban yuzeye projekte edebiliyoruz (o yuzeye cizebiliyoruz.)
float fShadowMatrix[16];


Texture lifebarTex("data/lifebar.shader");

App :: App() {
   // default oyun parametrelerini belirtiyoruz.
   // bu parametreler konfigurasyon dosyasindan girilen degerler ile
   // daha sonra degistirilebilir.
   gameParams.life = LIFE;
   gameParams.base_head_fork_time = BASE_HEAD_FORK_TIME;
   gameParams.max_trap_count = MAX_TRAP_COUNT;
   gameParams.trap_life_time = TRAP_LIFE_TIME;
   gameParams.fork_time_delta_scaling = FORK_TIME_DELTA_SCALING;
   gameParams.time_between_hbonus = TIME_BETWEEN_HBONUS;
   gameParams.time_between_dbonus = TIME_BETWEEN_HBONUS; 
}


// Bu fonksiyon lib3d framework tarafindan ilk once cagirilan fonksiyondur.
// Oyuna baslarken birkez yapilmasi gerekli olan genel ayarlamalar burada
// gerceklesir.
void App::onInit() {

   // mouse pointerini sakla
   SDL_WM_GrabInput(SDL_GRAB_ON);
   SDL_ShowCursor(0);
   
   // framework un bize sagladigi basit kamerayi konumlandir.
   simpleCamera.moveBackward(150);
   simpleCamera.moveUp(80);
   simpleCamera.turnDown(25);
   //setSimpleCameraSpeed(0.05f);

   // bazi opengl parametrelerini ayarliyoruz.
   glEnable(GL_COLOR_MATERIAL);
   glEnable(GL_LIGHTING);
   glEnable(GL_LIGHT0);
   glDisable(GL_TEXTURE_2D);
   glEnable(GL_NORMALIZE);
   glFrontFace(GL_CCW);

   // glfont sistemini baslat
   unsigned int tex_id;
   glGenTextures(1,&tex_id);
   glFontCreate(&font,"data/font.glf",tex_id);

   // golge icin kullanilan projeksiyon matrisini olustur.
   setShadowMatrix(fShadowMatrix, LightPosition, Plane);

   // basit opengl sis mekanizmasini aktif hale getir.
   float fogColor[] = {0.9f,0.1f,0.2f,1.0f};
   glFogi(GL_FOG_MODE, GL_EXP2);		// Fog Mode
   glFogfv(GL_FOG_COLOR, fogColor);			// Set Fog Color
   glFogf(GL_FOG_DENSITY, 0.0035f);				// How Dense Will The Fog Be
   glHint(GL_FOG_HINT, GL_DONT_CARE);			// Fog Hint Value
   //glFogf(GL_FOG_START,0.0f);				// Fog Start Depth
   //glFogf(GL_FOG_END, 100.0f);				// Fog End Depth
   //glEnable(GL_FOG);					// Enables GL_FOG
   glClearColor(fogColor[0],fogColor[1],fogColor[2],fogColor[3]);

   // hedef sembolunu olustur.
   target = new Target;
   
   damageTex = new Texture("data/damage.png");
   damageTex->build();
   terrTex = new Texture("data/detail2.jpg");
   //terrTex = new Texture("data/marble.bmp");
   terrTex->build();
   shake_screen = false;
   setState(App::GS_INTRO);
   timet = 0;  // oyunun baslangicindan itibaren gecen zamani burada tutuyoruz.

   // skorlari diskten oku
   loadScoresFromDisk();
   
   lifebarTex.build();
   intro.init();
   
   // gorsel arayuzu olustur
   buildUI();

   // ses sistemini baslat, arkaplan sesini devamli olarak caldir.
   soundman.init();
   soundman.startMenuBackgroundSound();

   srand(time(NULL));
   
   // ana zamanlayiciyi baslat.
   timer.start();
}

// Bu fonksiyon icerisinde tum arayuz sistemi olusturulmaktadir.
void App :: buildUI() {

   // arayuz sistemini baslat.
   ui.init();
   ui.setOwner(this);
   
   // ana menu icin kullanilan UIBanner nesnelerini olustur.
   UIBanner *banner1 = new UIBanner("data/basla_mi_us.png",0.0f,0.0f,0.4f,0.2f);
   UIBanner *banner2 = new UIBanner("data/basla_mi_s.shader",0.0f,0.0f,0.4f,0.2f);
   UIBanner *banner3 = new UIBanner("data/skor_mi_us.png",0.0f,-0.06f,0.4f,0.2f);
   UIBanner *banner4 = new UIBanner("data/skor_mi_s.shader",0.0f,-0.06f,0.4f,0.2f);
   UIBanner *banner5 = new UIBanner("data/hakkinda_mi_us.png",0.0f,-0.12f,0.4f,0.2f);
   UIBanner *banner6 = new UIBanner("data/hakkinda_mi_s.shader",0.0f,-0.12f,0.4f,0.2f);
   UIBanner *banner7 = new UIBanner("data/cikis_mi_us.png",0.0f,0.-0.18f,0.4f,0.2f);
   UIBanner *banner8 = new UIBanner("data/cikis_mi_s.shader",0.0f,-0.18f,0.4f,0.2f);
   
   // ana menuyu olustur ve menu elemanlarini ekle
   gmenu = ui.addGfxMenu();
   gmenu->addItem(banner1,banner2);
   gmenu->addItem(banner3,banner4);
   gmenu->addItem(banner5,banner6);
   gmenu->addItem(banner7,banner8);
   gmenu->setPos(0.45f,0.3f);

   // ana menude secim isleminde kullanilan belirteci yukle
   UIBanner *selectorbanner = new UIBanner("data/menuselector.shader",-0.06f,0.01f,0.08f,0.08f);
   gmenu->setSelectorBanner(selectorbanner);

   // oyunda zamani gostermek icin kullanilan label i yarat.
   timeLabel = ui.addLabel("",0.001f,0.999f);
   timeLabel->setVisible(false);
   
   // oyunda FPS gostermek icin kullanilan label i yarat.
   fpsLabel = ui.addLabel("",0.93f,0.999f);
   fpsLabel->setVisible(gameParams.showFps);
   
   // oyunda version gostermek icin kullanilan label i yarat.
   versionLabel = ui.addLabel("v"VERSION_STR,0.95f,0.03f);
   versionLabel->setVisible(true);
   
   // hakkinda kismi secilince goruntulenen yaziyi olustur ve satir satir bir diziye at.
   sprintf(about_lines[0],"+--+ Ruh Kapani  v%s +--+",VERSION_STR);
   sprintf(about_lines[1],"Programlama:             Mehmet Deniz Aydinoglu");
   sprintf(about_lines[2],"Grafik ve Modeller:      Cesitli serbest ve ticari kullanima kapali kaynaklar (teknik.txt)");
   sprintf(about_lines[3],"Gerceklestirim Tarihi:   Agustos-Ekim 2004");
   sprintf(about_lines[4],"Irtibat:                     deniz@oyunyapimi.org");
   sprintf(about_lines[5],"Selamlar:                  selamlar.txt dosyasina bakin...");
   sprintf(about_lines[6],"www.oyunyapimi.org - Turk Oyun Yapimcilarinin Bulusma Noktasi");
   
   // hakkinda panelinin bulunacagi pozisyon
   const float basex=0.01f, basey = 0.998f;
   
   // "hakkinda" kismini satir satir label nesneleri kullanarak olusturuyoruz.
   // bu label lari bir panel altinda topluyoruz.
   aboutPanel = ui.addPanel();
   aboutPanel->setPos(basex,basey);
   aboutPanel->addWidget(new UILabel(about_lines[0],0.35f,0.0f));
   aboutPanel->addWidget(new UILabel(about_lines[1],0.0f,-0.04f));
   aboutPanel->addWidget(new UILabel(about_lines[2],0.0f,-0.06f));
   aboutPanel->addWidget(new UILabel(about_lines[3],0.0f,-0.08f));
   aboutPanel->addWidget(new UILabel(about_lines[4],0.0f,-0.10f));
   aboutPanel->addWidget(new UILabel(about_lines[5],0.0f,-0.12f));
   aboutPanel->addWidget(new UILabel(about_lines[6],0.18f,-0.16f));
   
   aboutPanel->setVisible(false);

   // skor tablosunu da yine alt alta yerlesmis label lar ile olusturuyoruz
   // bu label lari bir panel altinda grupluyoruz.
   float sbasex = 0.01f + 0.4f;
   float sbasey = 0.998f - 0.03f;
   scoreNamesPanel = ui.addPanel();
   scoreNamesPanel->setPos(sbasex,sbasey);     
   scoreNamesPanel->addWidget(new UILabel("+--+ Skor Tablosu +--+",0,0));
   scoreNameLabels[0] = (UILabel*) scoreNamesPanel->addWidget(new UILabel("",0,-0.04f));
   scoreNameLabels[1] = (UILabel*) scoreNamesPanel->addWidget(new UILabel("",0,-0.06f));
   scoreNameLabels[2] = (UILabel*) scoreNamesPanel->addWidget(new UILabel("",0,-0.08f));
   scoreNameLabels[3] = (UILabel*) scoreNamesPanel->addWidget(new UILabel("",0,-0.10f));
   scoreNameLabels[4] = (UILabel*) scoreNamesPanel->addWidget(new UILabel("",0,-0.12f));
   scoreNameLabels[5] = (UILabel*) scoreNamesPanel->addWidget(new UILabel("",0,-0.14f));
   scoreNameLabels[6] = (UILabel*) scoreNamesPanel->addWidget(new UILabel("",0,-0.16f));
   scoreNameLabels[7] = (UILabel*) scoreNamesPanel->addWidget(new UILabel("",0,-0.18f));
   scoreNameLabels[8] = (UILabel*) scoreNamesPanel->addWidget(new UILabel("",0,-0.20f));
   scoreNameLabels[9] = (UILabel*) scoreNamesPanel->addWidget(new UILabel("",0,-0.22f));

   sbasex = 0.2f + 0.4f;
   sbasey = 0.978f - 0.03f;
   scoreScoresPanel = ui.addPanel();
   scoreScoresPanel->setPos(sbasex,sbasey);
   scoreScoreLabels[0] = (UILabel*) scoreScoresPanel->addWidget(new UILabel("",0,-0.02f));
   scoreScoreLabels[1] = (UILabel*) scoreScoresPanel->addWidget(new UILabel("",0,-0.04f));
   scoreScoreLabels[2] = (UILabel*) scoreScoresPanel->addWidget(new UILabel("",0,-0.06f));
   scoreScoreLabels[3] = (UILabel*) scoreScoresPanel->addWidget(new UILabel("",0,-0.08f));
   scoreScoreLabels[4] = (UILabel*) scoreScoresPanel->addWidget(new UILabel("",0,-0.10f));
   scoreScoreLabels[5] = (UILabel*) scoreScoresPanel->addWidget(new UILabel("",0,-0.12f));
   scoreScoreLabels[6] = (UILabel*) scoreScoresPanel->addWidget(new UILabel("",0,-0.14f));
   scoreScoreLabels[7] = (UILabel*) scoreScoresPanel->addWidget(new UILabel("",0,-0.16f));
   scoreScoreLabels[8] = (UILabel*) scoreScoresPanel->addWidget(new UILabel("",0,-0.18f));
   scoreScoreLabels[9] = (UILabel*) scoreScoresPanel->addWidget(new UILabel("",0,-0.20f));

   // devamli guncel tutulan skor ve isimler hs_name ve hs_score dizilerinde tutulur.
   // gerekli oldugu zaman ise burada yapildigi gibi ilgili label lardaki degerler 
   // bu dizi elemanlari ile guncellestirilir. Yani veriyi hem label lar kendi icinde
   // hemde biz dizi icinde tutuyoruz.
   for (int i=0;i<10;++i) {
      scoreNameLabels[i]->setText(hs_name[i]);
      scoreScoreLabels[i]->setText(hs_score[i]);
   }

   // skor guncelleme menusu olusturuluyor.
   UIBanner *banner00 = new UIBanner("data/sguncelle_mi_us.png",0.0f,0.0f,0.13f,0.06f);
   UIBanner *banner01 = new UIBanner("data/sguncelle_mi_s.shader",0.0f,0.0f,0.13f,0.06f);
   UIBanner *banner02 = new UIBanner("data/geridon_mi_us.png",0.0f,-0.02f,0.13f,0.06f);
   UIBanner *banner03 = new UIBanner("data/geridon_mi_s.shader",0.0f,-0.02f,0.13f,0.06f);
   
   sguncelleMenu = ui.addGfxMenu();
   sguncelleMenu->addItem(banner00,banner01);
   sguncelleMenu->addItem(banner02,banner03);
   sguncelleMenu->setPos(0.49f,0.7f);
   sguncelleMenu->setVisible(false);
      
   scoreNamesPanel->setVisible(false);
   scoreScoresPanel->setVisible(false);

   // yuksek skor yapildiginda aktif hale gecen "tebrikler" kismi olusturuluyor.
   // bu kisimda tebrikler yazisini iceren bir banner ve kullanicinin adini girebilecegi
   // bir UIEdit widget i bulunuyor.
   congratulationsPanel = ui.addPanel();
   congratulationsPanel->setPos(sbasex-0.2f,sbasey-0.25f); 
   congratulationsPanel->addWidget(new UIBanner("data/tebrikler.shader",0.0f,0.0f,0.4f,0.2f));
   congratulationsPanel->addWidget(new UILabel("Yuksek Skor Tablosuna Girmeye Hak Kazandiniz!",-0.08f,-0.10f));
   congratulationsPanel->addWidget(new UILabel("Lutfen Adinizi Girin:",0.03f,-0.12f));
   hsedit = (UIEdit*) congratulationsPanel->addWidget(new UIEdit());
   hsedit->setPos(0.03f,-0.16f);
   hsedit->setMaxChars(MAX_USER_NAME-1);
   congratulationsPanel->setVisible(false);

   // "Oyun Bitti" kismini goruntuleyen widgetlar olusturuluyor.
   nohighscorePanel = ui.addPanel();
   nohighscorePanel->setPos(sbasex-0.2f,sbasey-0.25f); 
   nohighscorePanel->addWidget(new UILabel("* Oyun Bitti *",0.06f,0));
   nohighscorePanel->addWidget(new UILabel("Yuksek Skor Tablosuna Giremediniz!",-0.06f+0.015f,-0.02f));
   nohighscorePanel->addWidget(new UILabel("Ana Menuye Donmek Icin Bir Tusa Basin",-0.075f+0.015f,-0.04f));
   nohighscorePanel->setVisible(false);

   // ana menu aktif widget olarak belirleniyor.
   ui.focusWidget(gmenu);

}

// Bu fonksiyon UIManager tarafindan herhangi bir event olustugunda
// ilgili event nesnesi ile birlikte cagirilir.
// Bir menu nesnesi elemani secildiginde, UIEdit nesnesi ise tusa basildigi
// yada giris tamamlandiginda bir event gonderebilir.
// Widget larin hangi kosullarda event gonderebildigini UI.cpp ve UI.h
// daki kodlara bakarak daha ayrintili bicimde inceleyebilirsiniz.
void App :: UIEvent_Callback(UIEvent *event) {
   
   // debug amacli.
   printf("EVENT: source: %s\n"
          "       UIEvent.data: %d\n",event->source->print(),event->data);

   // event in geldigi kaynaga gore ne yapilmasi gerektigini ilgili alt
   // fonksiyonlar belirliyor.   
   if (event->source == gmenu) {
      handle_mainMenuEvents(event);
   }
   else
   if (event->source == sguncelleMenu) {
      handle_sguncelleMenuEvents(event);
   }
   else
   if (event->source == hsedit) {
      handle_highscoreEvents(event);
   }
   else
   if (event->source == nohighscorePanel->getFocusedWidget()) {

      // bu panelden gelen herhangi bir event sonunda ana menuye donus yapilacak.

      setState(GS_INTRO);
      simpleCamera.set(&Position3d(&Vector3d(0,0,0),0,0,0));
      simpleCamera.moveBackward(150);
      simpleCamera.moveUp(80);
      simpleCamera.turnDown(25);
         
      nohighscorePanel->setVisible(false);
      scoreNamesPanel->setVisible(false);
      scoreScoresPanel->setVisible(false);

      ui.focusWidget(gmenu);
            
      soundman.stopInGameBackgroundSound();
      soundman.startMenuBackgroundSound();
      intro.restart();
   }



}

// Ana menu den gelen eventlara gore cesitli islemler yapiliyor.
void App :: handle_mainMenuEvents(UIEvent *event) {

   // secilen menu elemani degistiginde ses caldir.
   if (event->type == UIEvent::UIEVT_UIGFXMENU_ONITEM) {
      soundman.playMenuMove();
      return;
   }      
   
   // hangi elemanin secildigi event->data icerisinde var.
   unsigned int selectedItem = event->data;
      
   // secilen ana menu elemanina gore islem yaptiriliyor.
   switch (selectedItem) {
      case 3:    // cikis
         exit();
         break;
      case 0:    // oyuna basla

         setState(GS_INTRO_TO_GAME);
         ui.focusWidget(NULL);

         aboutPanel->setVisible(false);
         scoreNamesPanel->setVisible(false);
         scoreScoresPanel->setVisible(false);
         gmenu->setVisible(false);
         
         soundman.playBeginGameSpeech();
         intro.finish();

         break;

      case 1:    // yuksek skorlari goster.
         aboutPanel->setVisible(false);
         scoreNamesPanel->setVisible(!scoreNamesPanel->isVisible());
         scoreScoresPanel->setVisible(!scoreScoresPanel->isVisible());
         ui.focusWidget(sguncelleMenu);
         break;

      case 2:    // hakkinda kismini goster.
         aboutPanel->setVisible(!aboutPanel->isVisible());
         scoreNamesPanel->setVisible(false);
         scoreScoresPanel->setVisible(false);
         break;

   }

}

// Skor guncelleme menusunden gelen event lere gore cesitli islemler yapiliyor.
void App :: handle_sguncelleMenuEvents(UIEvent *event) {

   // secilen eleman degistiginde ses caldir.
   if (event->type == UIEvent::UIEVT_UIGFXMENU_ONITEM) {
      soundman.playMenuMove();
      return;
   }      
   
   // secilen elamani al.
   unsigned int selectedItem = event->data;
      
   switch (selectedItem) {
      case 0:  // skor guncelle
         {
            // aktif skor ve isimleri tScores yapisina kaydet
            tScores scores;
            for (int i=0;i<10;++i) {
               sprintf(scores.names[i],hs_name[i]);
               scores.scores[i] = hs_score[i];
            }
         
            // aktif skorlari oncelikle diske kaydet
            saveScoresToDisk();
            
            // aktif skorlari server uzerinden guncelle.
            // Server dan gelen guncel bilgileri yerel skorlara yansit.
            if (updateScores(scoreserver,scoreurl,&scores,1) == HTTP_NO_ERROR) {
               for (i=0;i<10;++i) {
                  strcpy(hs_name[i],scores.names[i]);
                  hs_score[i] = scores.scores[i];
                  scoreNameLabels[i]->setText(hs_name[i]);
                  scoreScoreLabels[i]->setText(hs_score[i]);
               }
            }
            else {
               message("baglanti hatasi!!");
            }
         }

         break;
      
      case 1:   // ana menuye don
         scoreNamesPanel->setVisible(false);
         scoreScoresPanel->setVisible(false);
         sguncelleMenu->setVisible(false);
         ui.focusWidget(gmenu);
         break;
   }

}

// Yuksek skor yapildiginda isim girme sirasinda kullanilan UIEdit
// widgeti tarafindan yollanan eventlar burada degerlendiriliyor.
void App :: handle_highscoreEvents(UIEvent *event) {

   // eger enter a basilarak giris tamamlanmis ise..
   if (event->type == UIEvent::UIEVT_UIEDIT_ENTERPRESSED) {
   
      // ilgili skoru ana skor tablosuna gecir ve goruntusunu guncelle.
      hs_score[hs_rank] = score;
      strcpy(hs_name[hs_rank],hsedit->getText());

      total_forked_head_count[hs_rank] = curr_total_forked_head_count;
      total_forked_bonus_count[hs_rank] = curr_total_forked_bonus_count;
      total_mouse_press[hs_rank] = curr_total_mouse_press;
         
      scoreScoreLabels[hs_rank]->setColor(1,1,1);
      scoreNameLabels[hs_rank]->setColor(1,1,1);
      scoreNamesPanel->setVisible(false);
      scoreScoresPanel->setVisible(false);
      congratulationsPanel->setVisible(false);
      hsedit->setVisible(false);
      hsedit->resetEnter();
      hsedit->setText("");

      // ana menuye don, oyunu baslangic durumuna getir.
      ui.focusWidget(gmenu);

      setState(GS_INTRO);
      simpleCamera.set(&Position3d(&Vector3d(0,0,0),0,0,0));
      simpleCamera.moveBackward(150);
      simpleCamera.moveUp(80);
      simpleCamera.turnDown(25);
      soundman.stopInGameBackgroundSound();
      soundman.startMenuBackgroundSound();

      intro.restart();
   }  
   else {
      // herhangi bir tusa basilarak giris yapilmaya devam ediyor.
      // ekrandaki giris goruntusunu tazele.
      scoreNameLabels[hs_rank]->setText(hsedit->getText());
   }

}

// Lib3d::BaseApp frameworku tarafindan herhangi bir tusa
// basilma durumunda otomatik olarak bu fonksiyon cagirilir.
void App :: onKeyPressed(const SDLKey &key) {

   // ESC ye basildiginda direk olarak programdan cik. Baska bir tusa basilmasi
   // halinde ise ilgili tusa basilma olayini UIMAnager a ilet.
   if (key == SDLK_ESCAPE)
      exit();
   else
      ui.keyEvent(key);  

}

// Oyun parametrelerini disaridan degistirmek icin kullanilir.
void App :: setGameParams(sGameParams params) {
   this->gameParams = params;
}

// Lib3d::BaseApp frameworku tarafindan devamli olarak
// cagirilan fonksiyon.
// Burada oyunun o anda bulundugu duruma gore ilgili alt yordamlar
// cagirliyor. Ayni zamanda ana zamanlayicinin degerlerinin alinmasi da
// burada oluyor.
void App::onRender() {

   static double dLastElapsed;
   double dElapsed;
   dElapsed = timer.elapsed();
   
   timer.start();
   timet += dElapsed;   // oyunun basindan beri gecen zaman.
   
   glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
   glLoadIdentity();

   // intro durumunda intro yu render et.
   if (state == GS_INTRO) {
      renderIntro(dElapsed);
   }
   else
   if (state == GS_INTRO_TO_GAME) {
      // oyuna giris durumunda introyu sonladir ve oyun durumuna gec.
      renderIntro(dElapsed);
      if (intro.finished() == true) {
         setState(GS_GAME);
         restartGame();       
      }
   }
   else {
      // oyun durumundayken oyunu render et.
      renderGame(dElapsed);
      soundman.playRandomMoaner();
   }
   
   // ui ve ses alt sistemlerinin update() fonksiyonlari cagiriliyor.
   soundman.update(dElapsed);
   ui.update(dElapsed);
   ui.render();

   // fps guncelleniyor..
   fpsLabel->setText(BaseApp::getFPS());
   
}

// intro durumunda iken intro nesnesi kullanilarak logo goruntulenmesi
// ve kurukafalarin gecis animasyonlari gerceklestiriliyor.
void App::renderIntro(float dt) {
   applySimpleCamera();
   intro.update(dt);
   intro.render();
}

// Ana oyun render isleminin gerceklestirildigi fonksiyon.
// Bu fonksiyonda ayni zamanda carpisma kontrolleri, kapan yaratma,
// hedef sembolu hareketi ve carpisma zamanindaki ekran titremesi gibi
// islerde hallediliyor.
// Oyun icerisindeki tum mantiksal dongunun bu fonksiyondan yonetildigini soyleyebiliriz.
void App::renderGame(float dt) {

   // kamera konumu ve acilarina gore ekrani ayarla.
   applySimpleCamera();

   // oyun sirasinda oyun mantigini calistir.
   // bu mantiga gore yeni dusmanlarin ve bonuslarin olusmasi kontrol edilecek.
   if (state == GS_GAME)
      runGameLogic(dt);

   // oyun biterken kameraya yere dogru cevirme animasyonu uygula
   if (state == GS_GAME_TO_END) {
      float dt = timet - game_end_begint;
      BaseApp::simpleCamera.moveDown(dt*0.05f);
      BaseApp::simpleCamera.turnDown(dt*0.06f);
      
      if (dt > 2) {

         // kamera donmesi bittiginde skor tablosunu goruntule.
         
         soundman.playEndGameSpeech();

         setState(GS_RENDER_SCORES);
         handleScoreTable();
      }

   }

   // eger bir carpma durumu varsa ekranda basit bir titreme efekti olustur.
   if (shake_screen) {
      float rndx = (rand()%10)*0.1 - 0.5;
      float rndy = (rand()%10)*0.1 - 0.5;
      glTranslatef(rndx,rndy,0);
   }

   // golge goruntulemek ile ilgili islemlere basliyoruz.
   // ekrana cizim ozelligini kapat. artik sadece depth buffer ve stencil buffer
   // gibi elemanlar etkilenecek.
   glColorMask(GL_FALSE, GL_FALSE, GL_FALSE, GL_FALSE);
   // hatta depth buffera olan etkileride kaldiriyoruz. Artik cizim isleminin
   // sonuclarindan sadece stencil buffer etkileniyor.
   glDepthMask(GL_FALSE);

   // stencil test islemini aktif hale getir.
   glEnable(GL_STENCIL_TEST);
   // cizim yaptigimiz her yer icin stencil buffer a 1 bilgisi yazilsin.
   glStencilFunc(GL_ALWAYS, 1, 0xFFFFFFFF);
   // o an stencil buffer da bir bilgi var ise bu bilgi 1 ile degistirilecek.
   // bu sayede stencil bufferda cizim olan bir bolge icin her zaman sadece 1
   // cizim olmayan bir bolge icin ise her zaman sadece 0 olacak.
   glStencilOp(GL_REPLACE, GL_REPLACE, GL_REPLACE);
				   
   // golgenin uzerine dusurulecegi zemin pixel processing icin karta gonderiliyor..
   // (hatirlatma: bu cizim isleminden sadece stencil buffer etkilenir!)
   // Bu sayede stencil buffer da golge cizimi yapilacak bolge belirtilmis oluyor.
   glPushMatrix();
   glRotatef(-90,1,0,0);
   const float sz = 500;
   glBegin(GL_QUADS);
   glVertex3f(-sz,-sz,0);
   glVertex3f(sz,-sz,0);
   glVertex3f(sz,sz,0);
   glVertex3f(-sz,sz,0);
   glEnd();
   glPopMatrix();

   // ekrana tekrar cizim yapabilmek icin gerekli buffer lar aciliyor.
   glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
   glDepthMask(GL_TRUE);
   // sadece stencil buffer da 1 olan yerlerde ekrana cizim yapilmasi isteniyor.
   // bu sayede golgeyi sadece taban alana cizecegiz.
   glStencilFunc(GL_EQUAL, 1, 0xFFFFFFFF);

   // stencil buffer in icerigi artik degismesin.
   glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);
   
   // taban alani simdi ciziyoruz.
   glDisable(GL_LIGHTING);
   glColor3ub(167,88,25);
   glPushMatrix();
   glRotatef(-90,1,0,0);
   //const float sz = 500;
   terrTex->begin();
   glBegin(GL_QUADS);
   glTexCoord2f(0,0); glVertex3f(-sz,-sz,0);
   glTexCoord2f(10,0); glVertex3f(sz,-sz,0);
   glTexCoord2f(10,10); glVertex3f(sz,sz,0);
   glTexCoord2f(0,10); glVertex3f(-sz,sz,0);
   glEnd();
   terrTex->end();
   glPopMatrix();
   glEnable(GL_LIGHTING);

   // target sembolunun pozisyonunu guncelliyor ve ekrana cizimini sagliyoruz.
   int mousex,mousey;
   SDL_GetRelativeMouseState(&mousex,&mousey);
   target->updateMouseCoordinates(mousex,mousey);
   Vector3d targetpos = target->getPosition();
   glPushMatrix();
   glTranslatef(targetpos.x,-8,targetpos.z);
   glDisable(GL_FOG);
   glDisable(GL_DEPTH_TEST);
   target->render();
   //glEnable(GL_FOG);
   glEnable(GL_DEPTH_TEST);
   glPopMatrix();  
   
   // trap listesinde bulunan tum kapan nesneleri icin update() ve render()
   // fonksiyonlari cagiriliyor. Ayrica PASSIVE duruma gecen trap nesneleri
   // siliniyor.
   list<Trap*>::iterator ti = trapList.begin();
   while (ti != trapList.end()) {
      Trap *trap = *ti;
      if (trap->getState() == Trap::TS_PASSIVE) {
         list<Trap*>::iterator crr  = ti;
         ++ti;
         trapList.erase(crr);
         delete trap;
         --trapCount;
      }
      else {
         trap->update(dt);
         trap->render();
         ++ti;
      }
      
      
   }

   // bonus listesinde bulunan tum bonus nesneleri icin update() ve render()
   // fonksiyonlari cagiriliyor.
   list<MovingObject*>::iterator it = bonusList.begin();
   while (it != bonusList.end()) {
      
      bool destroy_bonus = false;
      MovingObject *o = *it;
      
      // her bonusun bir trap e carpip carpmadigi burada kontrol ediliyor.
      list<Trap*>::iterator ti = trapList.begin();
      while (ti != trapList.end()) {
         
         // trap - bonus carpismasi var ise...
         if (o->checkCollision((*ti)->getPos(),10) == true) {
            
            // carpisan eger healthbonus ise..
            if (o->getType() == MovingObject::MOT_HBONUS) {
               
               HealthBonus *hb = (HealthBonus*) o;

               // bonus deactivated durumundan aktif duruma geciriliyor. (bonusu alma!)
               if (hb->getState() == HealthBonus::HBS_DEACTIVATED) {
                  hb->setState(HealthBonus::HBS_ACTIVATED);
                  soundman.playHealthBonusTaken();
               }
               else  // bonus alinmis durumda ise finished (olu) duruma geciriliyor.
               if (hb->getState() == HealthBonus::HBS_FINISHED) {
                  // olu bonus nesneleri daha sonra yok edilecek!
                  destroy_bonus = true;                  
               }

               // bonus alindigi icin enerjimiz dolu duruma getiriliyor.
               life = gameParams.life;
               
            }
            else   // carpisma destroyerbonus ile yapilmis ise..
            if (o->getType() == MovingObject::MOT_DBONUS) {
               
               DestroyerBonus *db = (DestroyerBonus*) o;
               if (db->getState() == DestroyerBonus::DBS_DEACTIVATED) {
               
                  // bu bonusu ilk olarak alma durumunda buraya geliniyor..
                  // tum kurukafalarin pozisyonlari addDestroyTarget()
                  // fonksiyonu ile bonusa gonderiliyor.
                  // Bonus kendi icinde bu pozisyonlara ates efekti koyacaktir.
                  list<Head*>::iterator hi = headList.begin();
                  while (hi != headList.end()) {
                     Head *h = *hi;
                  
                     h->setState(Head::HS_DYING);
                     h->setSpeed(0.05f);
                     Vector3d tpos = h->getPos();
                     tpos.y -= 15;
                     h->setTarget(tpos);

                     db->addDestroyTarget(h->getPos());
                     db->setState(DestroyerBonus::DBS_ACTIVATED);
                                                                              
                     ++hi;
                  }

                  soundman.playDestroyerBonusTaken();
                  
               }
               else  // isi biten bonusu daha sonra silecegiz!
               if (db->getState() == DestroyerBonus::DBS_FINISHED)
                  destroy_bonus = true;
            
            }

            // bu bonusu silmemiz gerekiyor.
            if (destroy_bonus) {
               list<MovingObject*>::iterator crr  = it;
               ++it;
               bonusList.erase(crr);
               delete o;
               o = NULL;
            }
            
            break;
         }
         ++ti;
      }     
      
      // bonus silinmemisse update() ve render() yordamlarini cagir.
      if (o) {
         o->update(dt);
         o->render();
         ++it;
      }
   }
   

   // tum Head nesneleri icin carpisma kontrolleri ve render() / update() islemleri
   // burada yapiliyor. Ayni zamanda golge de ciziliyor..
   glColor4f(1,1,1,1);
   list<Head*>::iterator hi = headList.begin();
   while (hi != headList.end()) {
      Head *head = *hi;
      Head::eHeadState hstate = head->getState();
      
		// kurukafayi goruntulemeden once golgesini cizecegiz.
      glPushMatrix();

      // golgeyi blend edecek sekilde texture olmadan ciz.
      glEnable(GL_BLEND);
      glBlendFunc(GL_ONE,GL_ONE_MINUS_SRC_ALPHA);

      // golge rengi.
      glColor4f(0.0, 0.0, 0.0, 0.1f);
      glDisable(GL_TEXTURE_2D);  // gerekli
      glDisable(GL_LIGHTING);    // gerekli 
      glDisable(GL_DEPTH_TEST);  // titreme (z-fighting) olmamsi icin gerekli
      glDisable(GL_FOG);         // gerekli

      // cizim yapilan yerde stencil buffer daki degerin 1 arttirilmasi gerekiyor.
      // bu durumda biz sadece stencil bufferda 1 olan yerlere golge cizecegimiz icin
      // boylelikle ustuste 2 yada daha fazla golge cizilmesini engelliyoruz.
      // az cizim yapmak icin guzel bir kucuk optimizasyon! Ayrica ustuste cizilen
      // golgeler blending den dolayi duzgun bir goruntu olusturmazlar.
      glStencilOp(GL_KEEP, GL_KEEP, GL_INCR);

      // golge matrisi icerisinde yer alan transformasyonlari aktif hale getir.
      // bu transformasyon sonucunda golge silueti taban yuzey ustunde goruntulenecek.
      glMultMatrixf(fShadowMatrix);
      glPushMatrix();
         
      // kafa siluetini golge olarak ciz.
      ((Head*)head)->renderSilhouette();
      glPopMatrix();

      // cizim icin kullandigimiz opengl durumuna geri geliyoruz.
      glEnable(GL_TEXTURE_2D);
      glEnable(GL_DEPTH_TEST);
      glDisable(GL_BLEND);
      glEnable(GL_LIGHTING);
      glEnable(GL_FOG);
      glPopMatrix();

      // stencil i kapat.
      glDisable(GL_STENCIL_TEST);

      // golge cizim islemi sona erdi..
      
      // Head hedefe ulasmis ise yoket.
      if (hstate == Head::HS_STOPPED || hstate == Head::HS_ATTARGET) {
         list<Head*>::iterator crr  = hi;
         ++hi;
         headList.erase(crr);
         delete head;

         // bize carpmis ise enerjimizi azalt ve ekrani salla.
         if (hstate == Head::HS_ATTARGET) {
            soundman.playPlayerPain();
            shake_screen = true;
            shake_time = timet;
            --life;
         }
      }
      else   // Head hareket halinde...
         if (hstate == Head::HS_MOVING) {
         // Trap - Head carpisma kontrolunu yap.
         list<Trap*>::iterator ti = trapList.begin();
         while (ti != trapList.end()) {
            Trap *trap = *ti;

            // Eger Head HT_AVOIDMAN turunde ise (kapandan kacan mavi kurukafa..)
            // ve kapana cok yaklasmis ise yeni bir waypoint belirlemesi gerekiyor.
            if (head->getType() == Head::HT_AVOIDMAN) {
               Vector3d tp = trap->getPos();
               Vector3d hp = head->getPos();
               float dist_sqr = (tp.x-hp.x)*(tp.x-hp.x) + (tp.y-hp.y)*(tp.y-hp.y) + (tp.z-hp.z)*(tp.z-hp.z);
               const float thold_sqr = 25.0f * 25.0f;               
               // eger kapana cok yakin degil ise kacabilir.
               if (dist_sqr < thold_sqr) {
                  Vector3d tpos = head->getTarget();
                  Vector3d wp(-tpos.x,tpos.y,-100.0f);
                  head->insertWaypoint(wp);
               }

            }

            // carpisma varmi?
            if (head->checkCollision(trap->getPos(),10) == true) {
               head->setState(Head::HS_DYING);
               head->setSpeed(0.05f);
               Vector3d tpos = trap->getPos();
               tpos.y -= 15;
               head->setTarget(tpos);
               soundman.playSkullTrapped();
            }
            ++ti;
         }

         head->update(dt);
         
         glPushMatrix();
         // kurukafayi normal olarak ciz.
         head->render();
         glPopMatrix();

         ++hi;
      }
      else
      if (hstate == Head::HS_DYING || hstate == Head::HS_DEPLOYING) {
         head->update(dt);
         head->render();
         ++hi;
      }

   }


   // ekranin sallanmasi durumunda hasar efektini ciz
   // (hadsar efekti aktif degil! - bug var ve guzel gozukmuyor :/
   if (shake_screen) {
      if (timet - shake_time < 0.2) {
         renderDamagedScreen();
      }
      else
         shake_screen = false;
   }

   // enerjimizi gosteren basit konsolu ciz.
   renderConsol();

   // oyun sonunda sis efektinin yok olmasi ile ilgili bir
   // hatayi onlemek icin :)
   glEnable(GL_FOG);

}

// Burada enerji durumumuzu gosteren basit konsol ciziliyor.
void App::renderConsol() {

   // Cizim 2 boyutlu bir projeksiyon kullanilarak direk olarak ekran yuzeyine
   // gelecek sekilde yapiliyor.
   glMatrixMode(GL_PROJECTION);
   glPushMatrix();
   glLoadIdentity();
   glOrtho(0.0, 1.0, 0.0, 1.0, -1.0, 1.0);
   //glOrtho(0.0, 800, 0.0, 600, -1.0, 1.0);

   glMatrixMode(GL_MODELVIEW);
   glLoadIdentity();
   
   glPushMatrix();
   float len = life * 0.1f;
   glEnable(GL_BLEND);
   glBlendFunc(GL_ONE,GL_ONE);
   glDisable(GL_LIGHTING);
   glColor4f(1,1,1,1);
   lifebarTex.begin();
   glBegin(GL_QUADS);
   glTexCoord2f(0,0); glVertex3f(0.0f,0.0f,0.0f);
   glTexCoord2f(life,0); glVertex3f(len,0.0f,0.0f);
   glTexCoord2f(life,1); glVertex3f(len,0.1f,0.0f);
   glTexCoord2f(0,1); glVertex3f(0.0f,0.1f,0.0f);
   glEnd();
   lifebarTex.end();
   glPopMatrix();
   glDisable(GL_BLEND);
   glEnable(GL_LIGHTING);

   glMatrixMode(GL_PROJECTION);
   glPopMatrix();
   
   glMatrixMode(GL_MODELVIEW);

}

// hasar durumunda ekran yuzeyine efekt cizen fonksiyon.
// Pekde guzel gozukmedigi icin kaldirdim!
void App::renderDamagedScreen() {
/*
   glMatrixMode(GL_PROJECTION);
   glPushMatrix();
   glLoadIdentity();
   glOrtho(0.0, 1.0, 0.0, 1.0, -1.0, 1.0);

   glMatrixMode(GL_MODELVIEW);
   glPushMatrix();
   glLoadIdentity();
   glEnable(GL_BLEND);
   glBlendFunc(GL_ONE,GL_ONE);
   
   glColor3f(0.1,0.1,0.1);

   glLoadIdentity();
   damageTex->begin();
   glBegin(GL_QUADS);
      damageTex->apply(0,1);
      glVertex2f(0,0);
      damageTex->apply(0,0);
      glVertex2f(1,0);
      damageTex->apply(1,0);
      glVertex2f(1,1);
      damageTex->apply(1,1);
      glVertex2f(0,1);
   glEnd();
   damageTex->end();

   glDisable(GL_BLEND);

   glPopMatrix();  // modelview
   
   glMatrixMode(GL_PROJECTION);
   glPopMatrix();
   
   glMatrixMode(GL_MODELVIEW);
*/
}


// mouse tusuna basildiginda cagirilan ve eklenebiliyor ise bir kapan yaratan rutin.
void App::addTrap() {

   Trap *trap = new Trap(gameParams.trap_life_time);
   trap->init();

   Vector3d tpos = target->getPosition();
   tpos.y = -20;
   trap->setPos(tpos);

   trapList.push_back(trap);

   soundman.playBuildTrap();

}

// Ne zaman kurukafa yaratilacagina, kurukafanin tipi ve gidecegi hedefe karar
// veren rutin. Ayni zamanda zamana bagli olarak gerekli bonus nesneleride burada
// yaratilmakta.
void App::runGameLogic(float dt) {

   float base_fork_time = gameParams.base_head_fork_time;   // seconds.
   float game_time = timet - game_start_time;
   
   float fork_time = base_fork_time - (game_time * gameParams.fork_time_delta_scaling);
   if (fork_time < 0.5f)
      fork_time = 0.5f;

   cftime -= dt;
   if (cftime <= 0) {
      
      cftime = fork_time;

      Head *head = new Head();
      head->init();
      //Vector3d pos(((rand()%5)-2)*30,-10,(-8+rand()%3)*30);
      Vector3d pos((rand()%250)-125,-10,(-8+rand()%3)*30);
      Vector3d tpos((rand()%108)-54,10,3*30);
      head->setPos(pos);
      head->addWaypoint(tpos);
      Head::eHeadType htype = (Head::eHeadType)(rand()%3);
      head->setType(htype);
      switch (htype) {
         case Head::HT_BONES:
         case Head::HT_GREEN:
         case Head::HT_BLACK:
            head->setSpeed(0.15f + htype*0.05f);
            break;
         case Head::HT_AVOIDMAN:  // kullanilmiyor!
            head->setSpeed(0.10f);
            break;

      }
      headList.push_back(head);

      ++curr_total_forked_head_count;

   }

   hbonus_ftime -= dt;
   if (hbonus_ftime < 0) {
      hbonus_ftime = gameParams.time_between_hbonus;
   
      HealthBonus *b = new HealthBonus;
      b->init();
      Vector3d pos(((rand()%5)-2)*30,0,(-2+rand()%3)*30);
      b->setPos(pos);
      bonusList.push_back(b);

      ++curr_total_forked_bonus_count;

   }

   dbonus_ftime -= dt;
   if (dbonus_ftime < 0) {
      dbonus_ftime = gameParams.time_between_dbonus;

      DestroyerBonus *b = new DestroyerBonus;
      b->init();
      Vector3d pos(((rand()%5)-2)*30,0,(-2+rand()%3)*30);
      b->setPos(pos);
      bonusList.push_back(b);

      ++curr_total_forked_bonus_count;

   }

   if (state == GS_GAME) {
      
      // oyun oynanma halinde ekranda gozuken zamani guncelle.
      score = timet - game_start_time;

      static char txt[16];
      sprintf(txt,"%.1f",score);
      timeLabel->setText(txt);
   }
      
   if (state != GS_GAME_TO_END) {
      if (life < 0) {
         if (state == GS_GAME) {
            game_end_begint = timet;
         }
         setState(GS_GAME_TO_END);
         shake_screen = true;
         shake_time = timet;

         soundman.playPlayerDeath();
      }
   }
   

}

// Skor tablosunu goruntuleme ve eger yuksek skor yapilmis ise bu tabloya giris
// yapilmasi islemleri buradan hallediliyor.
void App :: handleScoreTable() {

   int rank=-1;
   
   for (int i=0;i<10;++i) {     
      if (score >= hs_score[i]) {
         rank = i;
         break;
      }
   }

   ui.setVisible(true);

   if (rank != -1) {
      
      for (int j=9;j>rank;--j) {
         strcpy(hs_name[j],hs_name[j-1]);
         hs_score[j] = hs_score[j-1];
         scoreNameLabels[j]->setText(hs_name[j]);
         scoreScoreLabels[j]->setText(hs_score[j]);
         
      }
      
      ui.focusWidget(congratulationsPanel);

      scoreScoreLabels[rank]->setColor(1,0,0);
      scoreNameLabels[rank]->setColor(1,0,0);
      scoreNameLabels[rank]->setText("..........");
      scoreScoreLabels[rank]->setText(score);

      setState(GS_ENTER_NAME);
      hs_rank = rank;
   }
   else {
   
      setState(GS_NOHIGHSCORE);
      ui.focusWidget(nohighscorePanel);
   
   }
   
   scoreNamesPanel->setVisible(true);
   scoreScoresPanel->setVisible(true);
   
}

// Yenilme durumunda oyunun yeniden baslatilmasi burada yapiliyor.
void App :: restartGame() {

   trapCount = 0;
   game_start_time = timet;
   cftime = 0;
   life = gameParams.life;
   hbonus_ftime = gameParams.time_between_hbonus;
   dbonus_ftime = gameParams.time_between_dbonus;

   curr_total_forked_head_count = 0;
   curr_total_forked_bonus_count = 0;
   curr_total_mouse_press = 0;

   // tum kapanlari kaldir.
   list<Trap*>::iterator i = trapList.begin();
   while (i != trapList.end()) {
      Trap *trap = *i;
      list<Trap*>::iterator crr  = i;
      ++i;
      trapList.erase(crr);
      delete trap;
   }
      
   // tum bonuslari kaldir.
   list<MovingObject*>::iterator ii = bonusList.begin();
   while (ii != bonusList.end()) {
      MovingObject *o = *ii;
      list<MovingObject*>::iterator crr  = ii;
      ++ii;
      bonusList.erase(crr);
      delete o;
   }

   // varsa tum kurukafalari kaldir.
   list<Head*>::iterator iii = headList.begin();
   while (iii != headList.end()) {
      Head *h = *iii;
      list<Head*>::iterator crr  = iii;
      ++iii;
      headList.erase(crr);
      delete h;
   }  
   
   timeLabel->setVisible(true);

   soundman.stopMenuBackgroundSound();
   soundman.startInGameBackgroundSound();

}

// Skor tablosunu sifirlar.
void App::resetScores() {

   for (int i=0;i<10;++i) {
      sprintf(hs_name[i],"mda");
      hs_score[i] = 60.0f - 5.0f*i;
      total_forked_head_count[i]  = -1;
      total_forked_bonus_count[i] = -1;
      total_mouse_press[i]        = -1;
   }
}

// Skor tablosunu scores.dat dosyasindan okudugu degerler ile gunceller.
void App :: loadScoresFromDisk() {

   tScores scores;
   char buffer[2048];
   
   FILE *fp = fopen("scores.dat","rb");
   if (fp) {
      fread(buffer,2048,1,fp);
      fclose(fp);

      if (decodeScores(buffer,&scores)) {
         for (int i=0;i<10;++i) {
            sprintf(hs_name[i],scores.names[i]);
            hs_score[i] = scores.scores[i];
            total_forked_head_count[i]  = scores.int_array[i][0];
            total_forked_bonus_count[i] = scores.int_array[i][1];
            total_mouse_press[i]        = scores.int_array[i][2];
         }
      }
      else {
         message("scores.dat dosyasi bozulmus! skorlar sifirlandi!");
         resetScores();
      }

   }
   else {
      message("scores.dat dosyasi bulunamadi! skorlar sifirlandi!");
      resetScores();
   }  

}

// Skorlari scores.dat dosyasina kaydeder.
void App :: saveScoresToDisk() {

   char buffer[2048];
   tScores scores;

   for (int i=0;i<10;++i) {
      strcpy(scores.names[i],hs_name[i]);
      scores.scores[i] = hs_score[i];
   }
   
   encodeScores(buffer,&scores);
     
   FILE *fp = fopen("scores.dat","wb");
   fwrite(buffer,2048,1,fp);
   fclose(fp);

}

// Vertex koordinatlarini transforme edip istenilen yuzeye projeksiyonunu saglayan
// bir transformasyon matrisi burada olusturuluyor.
void App::setShadowMatrix(float fDestMat[16],const float fLightPos[4],const float fPlane[4]) {

   float dot;

   // dot product of plane and light position
   dot =	fPlane[0] * fLightPos[0] + 
         fPlane[1] * fLightPos[1] + 
         fPlane[1] * fLightPos[2] + 
         fPlane[3] * fLightPos[3];

   // first column
   fDestMat[0] = dot - fLightPos[0] * fPlane[0];
   fDestMat[4] = 0.0f - fLightPos[0] * fPlane[1];
   fDestMat[8] = 0.0f - fLightPos[0] * fPlane[2];
   fDestMat[12] = 0.0f - fLightPos[0] * fPlane[3];

   // second column
   fDestMat[1] = 0.0f - fLightPos[1] * fPlane[0];
   fDestMat[5] = dot - fLightPos[1] * fPlane[1];
   fDestMat[9] = 0.0f - fLightPos[1] * fPlane[2];
   fDestMat[13] = 0.0f - fLightPos[1] * fPlane[3];

   // third column
   fDestMat[2] = 0.0f - fLightPos[2] * fPlane[0];
   fDestMat[6] = 0.0f - fLightPos[2] * fPlane[1];
   fDestMat[10] = dot - fLightPos[2] * fPlane[2];
   fDestMat[14] = 0.0f - fLightPos[2] * fPlane[3];

   // fourth column
   fDestMat[3] = 0.0f - fLightPos[3] * fPlane[0];
   fDestMat[7] = 0.0f - fLightPos[3] * fPlane[1];
   fDestMat[11] = 0.0f - fLightPos[3] * fPlane[2];
   fDestMat[15] = dot - fLightPos[3] * fPlane[3];
}

// mesaj kutusu acan yardimci fonksiyon
static void message(const char *str) {

// kullanmiyoruz!
#ifdef __USE_MESSAGE_BOX__

#ifdef WIN32
   MessageBox(0,str,"Ruh Kapani",0);
#else
   // linux?
   fprintf(stdout,"%s\n",str);
#endif

#endif

}

