#include <lib3d/util2d/Texture.h>

#include <math.h>
#define  M_PI   3.14159265358979323846

Texture :: Texture(const char *path) {
   strcpy(this->path,path);
   enableMipmapping(true);
   stageCount = 0;
   stages = NULL;
   shaderTimer = new SimpleTimer;
   genTexFromData = false;
   id = 0;
   
   if (strcmp(strchr(this->path,'.'),".shader") == 0) {
      useShaders = true;
   }
   else
      useShaders = false;

   setWrapS(GL_REPEAT);
   setWrapT(GL_REPEAT);

}

Texture :: Texture(unsigned int id) {
   this->id = id;
   path[0] = '\0';
   enableMipmapping(true);
   stageCount = 0;
   stages = NULL;
   shaderTimer = new SimpleTimer;
   useShaders = false;
   genTexFromData = false;
   id = 0;

   setWrapS(GL_REPEAT);
   setWrapT(GL_REPEAT);
}

Texture :: Texture(unsigned char *data,int w,int h,int bpp,const char *name) {
   
   dataPtr  = data;
   data_w   = w;
   data_h   = h;
   data_bpp = bpp;
      
   genTexFromData = true;
   enableMipmapping(true);
   stageCount = 0;
   stages = NULL;
   shaderTimer = new SimpleTimer;
   useShaders = false;
   strcpy(path,name);
   id = 0;

   setWrapS(GL_REPEAT);
   setWrapT(GL_REPEAT);

}

Texture :: ~Texture() {
   
   if (shaderTimer)
      delete shaderTimer;
   if (stages)
      delete [] stages;

}

bool Texture :: isExists() {

   FILE *fp = fopen(path,"r");
   if (fp != NULL){
      fclose(fp);
      return true;
   }
   return false;

}

void Texture :: setShaderTimer(SimpleTimer *st) {
   if (shaderTimer)
      delete shaderTimer;
   shaderTimer = st;
}

Stage* Texture :: getStages() {
   return stages;
}

int Texture :: getStageCount() {
   return stageCount;
}

void Texture :: addStage(Stage *s) {
   
   if (stages) {
      Stage *ss = new Stage[stageCount+1];
      memcpy(ss,stages,stageCount*sizeof(Stage));
      stages[stageCount] = *s;
      stageCount++;
   }
   else {
      useShaders = true;
      stageCount = 2;
      stages = new Stage[stageCount];
      stages[0].texture = this;
      stages[1]=*s;
   }
   

}

void Texture :: enableMipmapping(bool b) {
   useMipMaps = b;
}

void Texture :: setWrapS(int ws) {
   wrapS = ws;
}

void Texture :: setWrapT(int wt) {
   wrapT = wt;
}

bool Texture :: build() {
   
   if (useShaders)
      return build_shaders();
   
   id = TextureManager::instance()->get(path);
   if (id != -1)
      return true;
      
   SDL_Surface *image1;
   
   if (genTexFromData) {
      static SDL_Surface sdl_surface;
      static SDL_PixelFormat sdl_pixelformat;
      sdl_surface.format = &sdl_pixelformat;
      
      image1 = &sdl_surface;
      image1->w = data_w;
      image1->h = data_h;
      image1->format->BytesPerPixel = data_bpp;
      image1->pixels = (unsigned char*) dataPtr;
   }
   else {
      image1 = load_using_SDL_image(path);
      if (!image1)
         return false;
   }

   // Create Texture	
   unsigned int t[1];
   glGenTextures(1, t);
   id = t[0];
   glBindTexture(GL_TEXTURE_2D, id);   // 2d texture (x and y size)

   glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, wrapS);
   glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, wrapT);

   int bpp = image1->format->BytesPerPixel;
   GLint pf = ((bpp==3) ? GL_RGB : GL_RGBA);

   if (useMipMaps) {
      glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR_MIPMAP_NEAREST);
      gluBuild2DMipmaps(GL_TEXTURE_2D,pf,image1->w,image1->h,pf,GL_UNSIGNED_BYTE,image1->pixels);   
   }
   else {
      glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR);
      glTexImage2D(GL_TEXTURE_2D, 0, bpp, image1->w, image1->h, 0, pf, GL_UNSIGNED_BYTE, image1->pixels);
   }
       
   if (!genTexFromData) {
      SDL_FreeSurface(image1);
   }

   TextureManager::instance()->add(path,id);

   return true;
}

// (Nate Miller in tga lib ini kullanarak Uncompressed TGA yukluyor...)
// Yada useShaders=true ise shader yukleniyor...
/*
bool Texture :: build() {

   if (useShaders)
      return build_shaders();
   
   glTga_t img;
   img.GenId(1,&id);

   if (useMipMaps) {
      img.SetMipMap(1);
      img.SetMinFilter(GL_LINEAR_MIPMAP_NEAREST);
   }

   img.SetWrapS(wrapS);
   img.SetWrapT(wrapT);

   if (img.Load(path,id) == 1) {
      img.Upload(1);
      return true;
   }

   return false;
}
*/

void Texture :: resetShaders() {

   for (int i=0;i<stageCount;++i) {
      stages[i].rot_degree = 0;
      stages[i].scale_sin_deg = 0;
      stages[i].scroll_s_curr = 0;
      stages[i].scroll_t_curr = 0;
      stages[i].rgb_sin_deg = 0;
      stages[i].anim_curr_frame = 0;
      stages[i].anim_curr_frame_time = 0;
   }
   
}

void Texture :: makeCurrent() {
   glEnable(GL_TEXTURE_2D);
   glBindTexture(GL_TEXTURE_2D,id);
}

void Texture :: begin() {
   
   register int i;
   float dt;
   
   dt = shaderTimer->timeElapsed() / 1000.0f;

   glActiveTextureARB(GL_TEXTURE0_ARB);
   makeCurrent();
      
   glMatrixMode(GL_TEXTURE);
   glLoadIdentity();
      
   for (i=0;i<stageCount;++i) {
      
      glActiveTextureARB(GL_TEXTURE0_ARB+i);
      glEnable(GL_TEXTURE_2D);
            
      if (stages[i].anim) {
         stages[i].anim_curr_frame_time += dt;
         if (stages[i].anim_curr_frame_time >= stages[i].anim_frame_secs) {
            if (++stages[i].anim_curr_frame >= stages[i].anim_frame_count)
               stages[i].anim_curr_frame = 0;
            stages[i].anim_curr_frame_time = 0;
         }
         glBindTexture(GL_TEXTURE_2D,stages[i].anim_textures[stages[i].anim_curr_frame]->getId());
      }
      else
         glBindTexture(GL_TEXTURE_2D,stages[i].texture->getId());
     
      glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, stages[i].tex_env_mode);
      glLoadIdentity();
      
      if (stages[i].detail) {
         glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_COMBINE_EXT);
         glTexEnvi(GL_TEXTURE_ENV, GL_RGB_SCALE_EXT, 2);
         glScalef(stages[i].detail_scale,stages[i].detail_scale,1);
      }
         
      if (stages[i].scroll) {
         stages[i].scroll_s_curr += stages[i].scroll_s * dt;
         stages[i].scroll_t_curr += stages[i].scroll_t * dt;
         glTranslatef(stages[i].scroll_s_curr,stages[i].scroll_t_curr,0);
      }
         
      if (stages[i].rotate) {
         glTranslatef(0.5,0.5,0);
         stages[i].rot_degree += stages[i].rot_speed * dt;
         glRotatef(stages[i].rot_degree,0,0,1);
         glTranslatef(-0.5,-0.5,0);
      }
                  
      if (stages[i].scale_sin) {
         stages[i].scale_sin_deg += 2*M_PI*dt/stages[i].scale_sin_period;
         float val = stages[i].scale_sin_base + stages[i].scale_sin_amp * sin(stages[i].scale_sin_deg);
         glTranslatef(0.5,0.5,0);
         glScalef(val,val,1);
         glTranslatef(-0.5,-0.5,0);
      }
          
      if (stages[i].rgb_sin) {
         stages[i].rgb_sin_deg += 2*M_PI*dt/stages[i].rgb_sin_period;
         float val_r = stages[i].rgb_sin_base_r + stages[i].rgb_sin_amp_r * sin(stages[i].rgb_sin_deg);
         float val_g = stages[i].rgb_sin_base_g + stages[i].rgb_sin_amp_g * sin(stages[i].rgb_sin_deg);
         float val_b = stages[i].rgb_sin_base_b + stages[i].rgb_sin_amp_b * sin(stages[i].rgb_sin_deg);
         glColor3f(val_r,val_g,val_b);
      }

      if (stages[i].envmapping) {
         glTexGeni(GL_S, GL_TEXTURE_GEN_MODE, GL_SPHERE_MAP);
         glTexGeni(GL_T, GL_TEXTURE_GEN_MODE, GL_SPHERE_MAP);
         glEnable(GL_TEXTURE_GEN_S);
         glEnable(GL_TEXTURE_GEN_T);
      }
      else {
         glDisable(GL_TEXTURE_GEN_S);
         glDisable(GL_TEXTURE_GEN_T);
      }
            
   }
   
   glMatrixMode(GL_MODELVIEW);


}

void Texture :: end() {
   
   register int i;
   
   if (useShaders) {
   
      for(i=0;i<stageCount;++i) {
         glActiveTextureARB(GL_TEXTURE0_ARB+i);
         glDisable(GL_TEXTURE_2D);
      }

      shaderTimer->update();
   }

   glActiveTextureARB(GL_TEXTURE0_ARB);
   glDisable(GL_TEXTURE_2D);

}

void Texture :: apply(float u,float v) {

   register int i;
   
   if (!useShaders) {
      glTexCoord2f(u,v);
   }
   else {
      for(i=0;i<stageCount;++i)
         glMultiTexCoord2fARB(GL_TEXTURE0_ARB+i,u,v);
   }

}

unsigned int Texture :: getId() {
   return id;
}

void Texture :: render() {

   begin();      

   glBegin(GL_QUADS);
            
      apply(1,0);
      glVertex2f(-0.5,-0.5);
            
      apply(0,0);
      glVertex2f( 0.5,-0.5);
            
      apply(0,1);
      glVertex2f( 0.5, 0.5);
            
      apply(1,1);
      glVertex2f(-0.5, 0.5);

   glEnd();

   end();
}

bool Texture :: build_shaders() {
   stages = ShaderLoader::load(path,&stageCount);
   return (stages != NULL);
}

SDL_Surface* Texture :: load_using_SDL_image(char *filename) {
    
   Uint8 *rowhi, *rowlo;
   Uint8 *tmpbuf, tmpch;
   SDL_Surface *image;
   int i, j;
   bool isTGA = false;

   strupr(filename);
   for (int fn=0;filename[fn];++fn) {
      if (filename[fn] == '.') {
         if (strcmp(filename+fn+1,"TGA") == 0) {
            isTGA = true;
         }
         break;
      }
   }

   image = IMG_Load(filename);
   if ( image == NULL ) {
       fprintf(stderr, "Unable to load %s: %s\n", filename, SDL_GetError());
       return(NULL);
   }
   /* GL surfaces are upsidedown and RGB, not BGR :-) */
   tmpbuf = (Uint8 *)malloc(image->pitch);
   if ( tmpbuf == NULL ) {
       fprintf(stderr, "Out of memory\n");
       return(NULL);
   }
   rowhi = (Uint8 *)image->pixels;
   rowlo = rowhi + (image->h * image->pitch) - image->pitch;
   int bpp = image->format->BytesPerPixel;
   for ( i=0; i<image->h/2; ++i ) {
       
      if (isTGA) {
         for ( j=0; j<image->w; ++j ) {
            tmpch = rowhi[j*bpp];
            rowhi[j*bpp] = rowhi[j*bpp+2];
            rowhi[j*bpp+2] = tmpch;
            tmpch = rowlo[j*bpp];
            rowlo[j*bpp] = rowlo[j*bpp+2];
            rowlo[j*bpp+2] = tmpch;
         }
      }

      memcpy(tmpbuf, rowhi, image->pitch);
      memcpy(rowhi, rowlo, image->pitch);
      memcpy(rowlo, tmpbuf, image->pitch);
      rowhi += image->pitch;
      rowlo -= image->pitch;
   }
   free(tmpbuf);

   return(image);
}
 
