#include <lib3d/terrain/terrain.h>

Terrain :: Terrain(int size,int step) {

   mapSize = size;
   setStepSize(step);
   setScale(1,1,1);
   texture = NULL;

   glFogi(GL_FOG_MODE, GL_LINEAR);
   glFogf(GL_FOG_START, 0.0);
   glFogf(GL_FOG_END, 50.0f);
   glFogi(GL_FOG_COORDINATE_SOURCE_EXT, GL_FOG_COORDINATE_EXT);

   setFogDepth(50.0f);
   setFogColor(0.8f,0.2f,0.2f);
   enableFog(false);

   map = new unsigned char[mapSize*mapSize];

}

Terrain :: ~Terrain() {
   delete []map;
}

void Terrain :: setScale(float x,float y,float z) {
   scale.set(x,y,z);
}

void Terrain :: setStepSize(int ss) {
   stepSize = ss;
}

void Terrain :: setFogDepth(float depth) {
   fogDepth = depth;
}

void Terrain :: setFogColor(float r,float g,float b) {
	float fg[4] = {r,g,b,1.0f};
   glFogfv(GL_FOG_COLOR, fg);
}

void Terrain :: enableFog(bool f) {

   fogEnabled = f;
   
   if (fogEnabled)
      glEnable(GL_FOG);
   else
      glDisable(GL_FOG);

}

bool Terrain :: isFogEnabled() {
   return fogEnabled;
}

void Terrain :: setTexture(Texture *t) {
   texture = t;
}

bool Terrain :: setTexture(char *path) {

   texture = new Texture(path);
   
   if (texture->isExists())
      return true;
   else {
      delete texture;
      texture = NULL;
      return false;
   }
}

void Terrain :: build(bool staticMap) {
 
   this->staticMap = staticMap;
   
   if (texture != NULL)
      texture->build();
   
   if (map != NULL && staticMap) {
   
      dispListId = glGenLists(1);
      glNewList(dispListId,GL_COMPILE);

      renderTerrain();

      glEndList();
   }

}

bool Terrain :: load(char *path) {

   FILE *fp = fopen(path,"rb");
   if (fp != NULL) {
      fread(map,1,mapSize*mapSize,fp);
      return true;
   }
   return false;

}

void Terrain :: generate(int octave_cnt,int noise_freq,int seed) {

   Perlin p(octave_cnt,noise_freq,50,seed);

   for (int i=0;i<mapSize;++i)
      for (int j=0;j<mapSize;++j)
         map[i*mapSize+j] = abs(p.Get((float)j/mapSize,(float)i/mapSize));

}

void Terrain :: render() const {
   if (staticMap)
      glCallList(dispListId);
   else
      renderTerrain();
}

float Terrain :: getHeightOnTerrain(Vector3d *pos) {

   Vector3d normal;
   float d;

   getPlaneOnTerrain(pos,&normal,&d);
      
   float x = pos->x + (mapSize/2)*scale.x;
   float z = pos->z + (mapSize/2)*scale.z;
            
   // plane denkleminden faydalanarak Y kordinatini hesapla..
   return (d-normal.x*x-normal.z*z)/normal.y;
      
}

float Terrain :: getHeightOnTerrain(Vector3d *pos,PlaneEqCache *cache) {
              
   float fgrid_i = pos->x/scale.x + mapSize/2;
   float fgrid_j = pos->z/scale.z + mapSize/2;
   int grid_i = (int) fgrid_i;
   int grid_j = (int) fgrid_j;
 
   int boundary = mapSize-2;
   if (grid_i > boundary) grid_i = boundary;
   if (grid_i < 0) grid_i = 0;
   if (grid_j > boundary) grid_j = boundary;
   if (grid_j < 0) grid_j = 0;
      
   float ii = fgrid_i - grid_i;
   float jj = fgrid_j - grid_j;
      
   // grid icindeki hangi ucgen plane i uzerinde bulunuyoruz..
   int triangle = 1;
   if (ii+jj >= 1) triangle = 2;
      
   // cache de uygun plane denklemi var mi?
   if (grid_i == cache->i &&
         grid_j == cache->j &&
            triangle == cache->triangle) {
         
      // camera pozisyonunu map uzeri pozisyona cevir..
      // camera(x,z) grid uzerinde (width/2)*scalex,(height/2)*scalez
      // pozisyonuna karsilik geliyor...
       
      float x = pos->x + ((mapSize/2)*scale.x);
      float z = pos->z + ((mapSize/2)*scale.z);
            
      return (cache->D-cache->normal.x*x-cache->normal.z*z)/cache->normal.y;
      
   }
      
   // cache de uygun normal ve D degerleri yoksa, bunlari hesapla..
   Vector3d p1(grid_i,map[(grid_j+1)*mapSize+grid_i],grid_j+1);
   Vector3d p2(grid_i+1,map[grid_j*mapSize+grid_i+1],grid_j);
   Vector3d p3(grid_i,map[grid_j*mapSize+grid_i],grid_j);
   Vector3d p4(grid_i+1,map[(grid_j+1)*mapSize+grid_i+1],grid_j+1);
   
   p1.scale(&scale);
   p2.scale(&scale);
   p3.scale(&scale);
   p4.scale(&scale);

   Vector3d v1,v2;
   if (triangle == 1) {
      v1.set(p2.x-p1.x,p2.y-p1.y,p2.z-p1.z);
      v2.set(p3.x-p1.x,p3.y-p1.y,p3.z-p1.z);
   }
   else {
      v1.set(p4.x-p1.x,p4.y-p1.y,p4.z-p1.z);
      v2.set(p2.x-p1.x,p2.y-p1.y,p2.z-p1.z);
   }
   
   cache->normal.set(&Vector3d::cross(&v1,&v2));
   cache->normal.normalize();
      
   float x = pos->x + (mapSize/2)*scale.x;
   float z = pos->z + (mapSize/2)*scale.z;
            
   cache->D = cache->normal.x*p1.x+
         cache->normal.y*p1.y+cache->normal.z*p1.z;
   
   // plane denkleminden faydalanarak Y kordinatini hesapla..
   float Y = (cache->D-cache->normal.x*x-cache->normal.z*z)/cache->normal.y;
      
   //kaydedilmemis degerleri cache e kaydet..
   cache->i = grid_i;
   cache->j = grid_j;
   cache->triangle = triangle;
      
   return Y;
}

float Terrain :: getSlopeOnTerrain(Vector3d *pos) {

   // XZ yuzey normali
   static const Vector3d xzNormal(0,1,0);

   float fgrid_i = pos->x/scale.x + mapSize/2;
   float fgrid_j = pos->z/scale.z + mapSize/2;
   int grid_i = (int) fgrid_i;
   int grid_j = (int) fgrid_j;
 
   int boundary = mapSize-2;
   if (grid_i > boundary) grid_i = boundary;
   if (grid_i < 0) grid_i = 0;
   if (grid_j > boundary) grid_j = boundary;
   if (grid_j < 0) grid_j = 0;
      
   float ii = fgrid_i - grid_i;
   float jj = fgrid_j - grid_j;
      
   // grid icindeki hangi ucgen plane i uzerinde bulunuyoruz..
   int triangle = 1;
   if (ii+jj >= 1) triangle = 2;
      
   // cache de uygun plane denklemi var mi?
   if (grid_i == cache.i &&
         grid_j == cache.j &&
            triangle == cache.triangle) {
         
      // Formul: |v1|*|v2|*sin(ang) = v1.v2
      return Vector3d::dot(&xzNormal,&cache.normal) / cache.normal.length();
   }
      
   // cache de uygun normal ve D degerleri yoksa Normali hesapla
   Vector3d p1(grid_i,map[(grid_j+1)*mapSize+grid_i],grid_j+1);
   Vector3d p2(grid_i+1,map[grid_j*mapSize+grid_i+1],grid_j);
   Vector3d p3(grid_i,map[grid_j*mapSize+grid_i],grid_j);
   Vector3d p4(grid_i+1,map[(grid_j+1)*mapSize+grid_i+1],grid_j+1);
      
   p1.scale(&scale);
   p2.scale(&scale);
   p3.scale(&scale);
   p4.scale(&scale);

   Vector3d v1,v2;
   if (triangle == 1) {
      v1.set(p2.x-p1.x,p2.y-p1.y,p2.z-p1.z);
      v2.set(p3.x-p1.x,p3.y-p1.y,p3.z-p1.z);
   }
   else {
      v1.set(p4.x-p1.x,p4.y-p1.y,p4.z-p1.z);
      v2.set(p2.x-p1.x,p2.y-p1.y,p2.z-p1.z);
   }
   
   //kaydedilmemis degerleri cache e kaydet..
   cache.normal.set(&Vector3d::cross(&v1,&v2));
   cache.normal.normalize();
   cache.D = cache.normal.x*p1.x+cache.normal.y*p1.y+cache.normal.z*p1.z;
   cache.i = grid_i;
   cache.j = grid_j;
   cache.triangle = triangle;

   // Formul: |v1|*|v2|*sin(ang) = v1.v2
   return Vector3d::dot(&xzNormal,&cache.normal) / cache.normal.length();
}

void Terrain :: getPlaneOnTerrain(const Vector3d *pos,Vector3d *n,float *d) {

   float fgrid_i = pos->x/scale.x + mapSize/2;
   float fgrid_j = pos->z/scale.z + mapSize/2;
   int grid_i = (int) fgrid_i;
   int grid_j = (int) fgrid_j;
 
   int boundary = mapSize-2;
   if (grid_i > boundary) grid_i = boundary;
   if (grid_i < 0) grid_i = 0;
   if (grid_j > boundary) grid_j = boundary;
   if (grid_j < 0) grid_j = 0;
      
   float ii = fgrid_i - grid_i;
   float jj = fgrid_j - grid_j;
      
   // grid icindeki hangi ucgen plane i uzerinde bulunuyoruz..
   int triangle = 1;
   if (ii+jj >= 1) triangle = 2;
      
   Vector3d p1(grid_i,map[(grid_j+1)*mapSize+grid_i],grid_j+1);
   Vector3d p2(grid_i+1,map[grid_j*mapSize+grid_i+1],grid_j);
   Vector3d p3(grid_i,map[grid_j*mapSize+grid_i],grid_j);
   Vector3d p4(grid_i+1,map[(grid_j+1)*mapSize+grid_i+1],grid_j+1);
   
   p1.scale(&scale);
   p2.scale(&scale);
   p3.scale(&scale);
   p4.scale(&scale);

   Vector3d v1,v2;
   if (triangle == 1) {
      v1.set(p2.x-p1.x,p2.y-p1.y,p2.z-p1.z);
      v2.set(p3.x-p1.x,p3.y-p1.y,p3.z-p1.z);
   }
   else {
      v1.set(p4.x-p1.x,p4.y-p1.y,p4.z-p1.z);
      v2.set(p2.x-p1.x,p2.y-p1.y,p2.z-p1.z);
   }
   
   n->set(&Vector3d::cross(&v1,&v2));
   n->normalize();
   *d = n->x*p1.x+n->y*p1.y+n->z*p1.z;

}

void Terrain :: renderTerrain() const {

   int X,Y;       						// Create some variables to walk the array with.
   int x, y, z;							// Create some variables for readability
   bool bSwitchSides = false;

   glPushMatrix();
   glScalef(scale.x,scale.y,scale.z);
   glTranslatef(-(float)mapSize/2,0,-(float)mapSize/2);

   if (texture)
      texture->begin();

   // We want to render triangle strips
   glBegin( GL_TRIANGLE_STRIP );			

   // Go through all of the rows of the height map
   for ( X = 0; X < mapSize; X+=stepSize ) {
      
      // Check if we need to render the opposite way for this column
      if(bSwitchSides) {	
         
         // Render a column of the terrain, for this current X.
         // We start at MAP_SIZE and render down to 0.
         for ( Y = mapSize-stepSize; Y >= 0; Y-=stepSize ) {

            // Get the (X, Y, Z) value for the bottom left vertex		
            x = X;							
            y = map[Y*mapSize+X];	
            z = Y;							

            setFogCoord(y);
            
            // Set the current texture coordinate and render the vertex
            setTextureCoord( (float)x, (float)z );
            glVertex3i(x, y, z);		

            // Get the (X, Y, Z) value for the bottom right vertex		
            x = X + stepSize; 
            y = map[Y*mapSize+x]; 
            z = Y;

            setFogCoord(y);

            // Set the current texture coordinate and render the vertex
            setTextureCoord( (float)x, (float)z );
            glVertex3i(x, y, z);			
         }
      }
      else {	

         // Render a column of the terrain, for this current X.
         // We start at 0 and render down up to MAP_SIZE.
         
         for ( Y = 0; Y < mapSize; Y+=stepSize ) {

            // Get the (X, Y, Z) value for the bottom right vertex		
            x = X + stepSize; 
            y = map[Y*mapSize+x]; 
            z = Y;

            setFogCoord(y);

            // Set the current texture coordinate and render the vertex
            setTextureCoord( (float)x, (float)z );
            glVertex3i(x, y, z);

            // Get the (X, Y, Z) value for the bottom left vertex		
            x = X;							
            y = map[Y*mapSize+X];	
            z = Y;							

            setFogCoord(y);
            
            // Set the current texture coordinate and render the vertex
            setTextureCoord( (float)x, (float)z );
            glVertex3i(x, y, z);		
         }
      }

      // Switch the direction the column renders to allow the fluid tri strips
      bSwitchSides = !bSwitchSides;
   }

   // Stop rendering triangle strips
   glEnd();

   if (texture)
      texture->end();

   glPopMatrix();

}

void Terrain :: setTextureCoord(float x, float z) const {
	
   // Find the (u, v) coordinate for the current vertex
	float u =  (float)x / (float) mapSize;
	float v = -(float)z / (float) mapSize;
	
	// Give OpenGL the current terrain texture coordinate for our height map
	glMultiTexCoord2fARB(GL_TEXTURE0_ARB, u, v);

	// Give OpenGL the current detail texture coordinate for our height map
	glMultiTexCoord2fARB(GL_TEXTURE1_ARB, u, v);

}

void Terrain :: setFogCoord(float y) const {

	// Check if the height of this vertex is greater than the depth (needs no fog)
	if(y > fogDepth)
      glFogCoordfEXT(0);
	// Otherwise, calculate the fog depth for the current vertex
	else
      glFogCoordfEXT(fogDepth-y);

}




