#include <lib3d/bsp/Q3BspRenderer.h>

Q3BspRenderer :: Q3BspRenderer() {

   vertices = NULL;
   faces    = NULL;
   planes   = NULL;
   nodes    = NULL;
   leafs    = NULL;
   brushes  = NULL;
   brushSides = NULL;
   visdata.pBitsets = NULL;
   leafFaces = NULL;
   leafBrushes = NULL;

   setScale(1.0f);
   enablePVS(true);
   currCluster = -1;
   faceRendered = 0;
   
   memset(textures,0,sizeof(textures));
   memset(lightmaps,0,sizeof(lightmaps));
   strcpy(texpath,".\\");


}

Q3BspRenderer :: ~Q3BspRenderer() {
   
   if (vertices)
      delete [] vertices;

   if (faces)
      delete [] faces;

   if (planes)
      delete [] planes;

   if (nodes)
      delete [] nodes;

   if (leafs)
      delete [] leafs;

   if (leafFaces)
      delete [] leafFaces;

   if (leafBrushes)
      delete [] leafBrushes;
   
   if (brushSides)
      delete [] brushSides;

   if (brushes)
      delete [] brushes;

   for (int i=0;i<MAX_TEXTURE_COUNT;++i) {
      if (textures[i])
         delete textures[i];
      if (lightmaps[i])
         delete lightmaps[i];
   }
  
}

void Q3BspRenderer :: setTexturePath(const char *path) {
   sprintf(texpath,"%s\\",path);
}

int Q3BspRenderer :: load(const char *bspfile) {

   int status = Q3BSP_NOERROR;
   
   // open bsp file
   FILE *fp = fopen(bspfile,"rb");
   if (!fp)
      return Q3BSP_ERR_FILE_NOT_FOUND;

   // read bsp header
   q3bsp_tBSPHeader header;
   fread(&header,1,sizeof(q3bsp_tBSPHeader),fp);

   // check if it is a BSP file
   if (memcmp(header.strID,"IBSP",4) != 0)
      return Q3BSP_ERR_BADID;
   
   // check version (quake3 uses version 0x2e)
   if (header.version != 0x2e)
      return Q3BSP_ERR_NOT_SUPPORTED_VERSION;
   
   // read lump directory
   q3bsp_tBSPLump lumps[kMaxLumps];
   fread(lumps,kMaxLumps,sizeof(q3bsp_tBSPLump),fp);

   // load textures
   int textureCount = lumps[kTextures].length / sizeof(q3bsp_tBSPTexture);
      
   fseek(fp,lumps[kTextures].offset,SEEK_SET);
   for (int i=0;i<textureCount;++i) {
      q3bsp_tBSPTexture tex;
      
      // read next texture definition from bsp file
      fread(&tex,1,sizeof(q3bsp_tBSPTexture),fp);

      // search, find and load the texture
      if (!searchAndLoadTexture(i,&tex)) {
         status =  Q3BSP_ERR_MISSING_TEXTURE;
         unsigned char buffer[16*16*3];
         memset(buffer,0,sizeof(buffer));
         textures[i] = new Texture(buffer,16,16,3,"_blank");
         textures[i]->build();
      }
   }

   // read planes
   planeCount = lumps[kPlanes].length / sizeof(q3bsp_tBSPPlane);
   planes = new q3bsp_tBSPPlane[planeCount];
   fseek(fp,lumps[kPlanes].offset,SEEK_SET);
   fread(planes,planeCount,sizeof(q3bsp_tBSPPlane),fp);

   // read nodes
   nodeCount = lumps[kNodes].length / sizeof(q3bsp_tBSPNode);
   nodes = new q3bsp_tBSPNode[nodeCount];
   fseek(fp,lumps[kNodes].offset,SEEK_SET);
   fread(nodes,nodeCount,sizeof(q3bsp_tBSPNode),fp);

   // read leafs
   leafCount = lumps[kLeafs].length / sizeof(q3bsp_tBSPLeaf);
   leafs = new q3bsp_tBSPLeaf[leafCount];
   fseek(fp,lumps[kLeafs].offset,SEEK_SET);
   fread(leafs,leafCount,sizeof(q3bsp_tBSPLeaf),fp);

   // read faces
   faceCount = lumps[kFaces].length / sizeof(q3bsp_tBSPFace);
   faces = new q3bsp_tBSPFace[faceCount];
   fseek(fp,lumps[kFaces].offset,SEEK_SET);
   fread(faces,faceCount,sizeof(q3bsp_tBSPFace),fp);

   // read leaf faces
   leafFaceCount = lumps[kLeafFaces].length / sizeof(int);
   leafFaces = new int[leafFaceCount];
   fseek(fp,lumps[kLeafFaces].offset,SEEK_SET);
   fread(leafFaces,leafFaceCount,sizeof(int),fp);

   // setup face drawing control vector
   faceDrawStatus.Resize(leafFaceCount);
   faceDrawStatus.ClearAll();

   // read vertices
   vertexCount = lumps[kVertices].length / sizeof(q3bsp_tBSPVertex);
   vertices = new q3bsp_tBSPVertex[vertexCount];
   fseek(fp,lumps[kVertices].offset,SEEK_SET);
   fread(vertices,vertexCount,sizeof(q3bsp_tBSPVertex),fp);
     
   // read brushes
   brushCount = lumps[kBrushes].length / sizeof(q3bsp_tBSPBrush);
   brushes = new q3bsp_tBSPBrush[brushCount];
   fseek(fp,lumps[kBrushes].offset,SEEK_SET);
   fread(brushes,brushCount,sizeof(q3bsp_tBSPBrush),fp);
     
   // read brushSides
   brushSideCount = lumps[kBrushSides].length / sizeof(q3bsp_tBSPBrushSide);
   brushSides = new q3bsp_tBSPBrushSide[brushSideCount];
   fseek(fp,lumps[kBrushSides].offset,SEEK_SET);
   fread(brushSides,brushSideCount,sizeof(q3bsp_tBSPBrushSide),fp);

   // read leaf brushes
   leafBrushCount = lumps[kLeafBrushes].length / sizeof(int);
   leafBrushes = new int[leafBrushCount];
   fseek(fp,lumps[kLeafBrushes].offset,SEEK_SET);
   fread(leafBrushes,leafBrushCount,sizeof(int),fp);
     
     
   // read visdata
   if (lumps[kVisData].length>0) {
      fseek(fp,lumps[kVisData].offset,SEEK_SET);
      fread(&(visdata.numOfClusters),1,sizeof(int), fp);
      fread(&(visdata.bytesPerCluster),1,sizeof(int), fp);
      int size = visdata.numOfClusters * visdata.bytesPerCluster;
      visdata.pBitsets = new unsigned char[size];
      fread(visdata.pBitsets,1,sizeof(unsigned char)*size,fp);
   }
   else
      visdata.pBitsets = NULL;

   // read lightmaps
   int lightmapCount = lumps[kLightmaps].length / sizeof(q3bsp_tBSPLightmap);
   q3bsp_tBSPLightmap *maps = new q3bsp_tBSPLightmap[lightmapCount];
   fseek(fp,lumps[kLightmaps].offset,SEEK_SET);
   fread(maps,lightmapCount,sizeof(q3bsp_tBSPLightmap),fp);

   // build lightmap textures from raw lightmap image data
   for (i=0;i<lightmapCount;++i) {

      // increase lightmap texture gamma
      for (int row=0;row<128;++row)
         for (int col=0;col<128;++col) {
		      
            float scale = 1.0f, temp = 0.0f;
		      float r = 0, g = 0, b = 0;
            const float GAMMA_FACTOR = 10.0f;

            // extract the current RGB values
            r = (float)maps[i].imageBits[row][col][0];
		      g = (float)maps[i].imageBits[row][col][1];
		      b = (float)maps[i].imageBits[row][col][2];

		      // Multiply the GAMMA_FACTOR by the RGB values, while keeping it to a 255 ratio
		      r = r * GAMMA_FACTOR / 255.0f;
		      g = g * GAMMA_FACTOR / 255.0f;
		      b = b * GAMMA_FACTOR / 255.0f;
		      
		      // Check if the the values went past the highest value
		      if(r > 1.0f && (temp = (1.0f/r)) < scale) scale=temp;
		      if(g > 1.0f && (temp = (1.0f/g)) < scale) scale=temp;
		      if(b > 1.0f && (temp = (1.0f/b)) < scale) scale=temp;

		      // Get the scale for this pixel and multiply it by our pixel values
		      scale*=255.0f;		
		      r*=scale;	g*=scale;	b*=scale;

		      // Assign the new gamma'nized RGB values to our image
		      maps[i].imageBits[row][col][0] = (byte)r;
		      maps[i].imageBits[row][col][1] = (byte)g;
		      maps[i].imageBits[row][col][2] = (byte)b;
         }
         
      static int lightMap_id=0;
      char name[256];
      sprintf(name,"lightmap_%d",lightMap_id);
      lightMap_id++;
         
      lightmaps[i] = new Texture((unsigned char*)(maps[i].imageBits),128,128,3,name);
      lightmaps[i]->build();

   }
   delete [] maps;
      
   for (i=0;i<vertexCount;++i) {

      // swap z and y axes.. negate y axe..
      float temp = vertices[i].vPosition[1];
      vertices[i].vPosition[1] = vertices[i].vPosition[2];
      vertices[i].vPosition[2] = -temp;

      // negate v tex coordinate
      vertices[i].vTextureCoord[1] *= -1;
      
      // scale vertices
      vertices[i].vPosition[0] *= levelScale;
      vertices[i].vPosition[1] *= levelScale;
      vertices[i].vPosition[2] *= levelScale;
   }

   for (i=0;i<planeCount;++i) {
      // swap z and y axes.. negate y axe..
      float temp = planes[i].vNormal[1];
      planes[i].vNormal[1] = planes[i].vNormal[2];
      planes[i].vNormal[2] = -temp;

   }

   for (i=0;i<leafCount;++i) {
      // change bounding box coordinates to match our opengl implementation..
      // swap z and y axes.. negate y axe..
      float temp = leafs[i].min[1];
      leafs[i].min[1] = leafs[i].min[2];
      leafs[i].min[2] = -temp;

      temp = leafs[i].max[1];
      leafs[i].max[1] = leafs[i].max[2];
      leafs[i].max[2] = -temp;
   }

   fclose(fp);
   return status;
}

// texture name in texture lump does not have any file extension.
// because of this we gonna try some common extension like jpg and tga
bool Q3BspRenderer :: searchAndLoadTexture(int idx,q3bsp_tBSPTexture *tex) {
   
   char name_jpg[256],name_tga[256];
   
   // guess texture extension as .jpg
   strcpy(name_jpg,texpath);
   strcat(name_jpg,tex->strName);
   strcat(name_jpg,".jpg");

   // try to load texture
   textures[idx] = new Texture(name_jpg);
   if (textures[idx]->isExists()) {
      // texture found. build it.
      textures[idx]->build();
      fprintf(stdout,"Q3BspRenderer:: texture loaded: %s\n",name_jpg);
      return true;
   }
   delete textures[idx];
   
   // guess texture extension as .tga
   strcpy(name_tga,texpath);
   strcat(name_tga,tex->strName);
   strcat(name_tga,".tga");

   // try to load texture
   textures[idx] = new Texture(name_tga);
   if (textures[idx]->isExists()) {
      // texture found. build it.
      textures[idx]->build();
      fprintf(stdout,"Q3BspRenderer:: texture loaded: %s\n",name_tga);
      return true;
   }
   delete textures[idx];
   
   // texture not found.. or its a shader file
   // set its texture to a default one
   sprintf(name_jpg,"%s\\%s",texpath,SHADER_TEXTURE);
   textures[idx] = new Texture(name_jpg);
   if (textures[idx]->isExists()) {
      fprintf(stdout,"Q3BspRenderer:: shader tex applied!\n");
      textures[idx]->build();
      return true;
   }
   delete textures[idx];
   
   // no texture file found
   fprintf(stdout,"Q3BspRenderer:: no tex found!\n");
   return false;
}

q3bsp_tBSPLeaf* Q3BspRenderer :: findLeaf(const Vector3d *eyePos) {
   
   int n = 0;
   
   while (1) {
      
      // if current node points to a leaf than we have found our cluster
      if (n < 0) {
         q3bsp_tBSPLeaf *leaf = &leafs[~n];
         return leaf;
      }
      
      q3bsp_tBSPNode  *node = &nodes[n];
      q3bsp_tBSPPlane *plane = &planes[node->plane];

      int side = plane->vNormal[0]*eyePos->x + 
                  plane->vNormal[1]*eyePos->y +
                   plane->vNormal[2]*eyePos->z - plane->d;
      
      if (side >= 0)
         n = node->front;
      else
         n = node->back;

   }


}

void Q3BspRenderer :: render(const Vector3d *eyePos) {

   frustum.CalculateFrustum();
   faceDrawStatus.ClearAll();
   faceRendered = 0;
   
   glVertexPointer(3,GL_FLOAT,sizeof(q3bsp_tBSPVertex),&(vertices[0].vPosition));

   glClientActiveTextureARB(GL_TEXTURE0_ARB);
   glTexCoordPointer(2, GL_FLOAT, sizeof(q3bsp_tBSPVertex), &(vertices[0].vTextureCoord));

   glClientActiveTextureARB(GL_TEXTURE1_ARB);
   glTexCoordPointer(2, GL_FLOAT, sizeof(q3bsp_tBSPVertex), &(vertices[0].vLightmapCoord));

   glEnableClientState(GL_VERTEX_ARRAY);
   glEnableClientState(GL_TEXTURE_COORD_ARRAY);
	
   glClientActiveTextureARB(GL_TEXTURE0_ARB);
   glEnableClientState(GL_TEXTURE_COORD_ARRAY);

   glClientActiveTextureARB(GL_TEXTURE1_ARB);
   glEnableClientState(GL_TEXTURE_COORD_ARRAY);

   // find the cluster we are in and get leafs clustet
   int cluster = findLeaf(eyePos)->cluster;
   currCluster = cluster;
   
   // loop through all the leafs and render if they are seen
   // from our cluster (PVS...)
   for (int i=0;i<leafCount;++i) {

      q3bsp_tBSPLeaf *targetf = &leafs[i];
      
      int visible = 1;
      if (visdata.pBitsets && pvsEnabled && cluster>0) {
         unsigned char visSet = visdata.pBitsets[(cluster*visdata.bytesPerCluster) + (targetf->cluster / 8)];
         visible = visSet & (1 << ((targetf->cluster) & 7));
      }

      if (visible) {

         // now test leafs bounding box for a frustum hit
         if (frustum.BoxInFrustum(targetf->min[0],targetf->min[1],targetf->min[2],
              targetf->max[0],targetf->max[1],targetf->max[2])) {
         
            for (int ff=0;ff<targetf->numOfLeafFaces;++ff) {
               
               // get the face index
               int face_idx = leafFaces[targetf->leafface+ff];

               // draw the face if its not rendered before
               if (!faceDrawStatus.On(face_idx)) {
                          
                  faceDrawStatus.Set(face_idx);
                  faceRendered++;

                  q3bsp_tBSPFace *f = &faces[face_idx];
                  if (f->type == FACE_POLYGON) {

                     // render the face
                     glActiveTextureARB(GL_TEXTURE0_ARB);
                     glEnable(GL_TEXTURE_2D);
                     glBindTexture(GL_TEXTURE_2D,textures[f->textureID]->getId());
      
                     glActiveTextureARB(GL_TEXTURE1_ARB);
                     glEnable(GL_TEXTURE_2D);
                     if (f->lightmapID != -1)
                        glBindTexture(GL_TEXTURE_2D,lightmaps[f->lightmapID]->getId());
         
                     glDrawArrays(GL_TRIANGLE_FAN,f->startVertIndex,f->numOfVerts);

                  }
               }

            }
         
         }

      }

   }

   //glDisableClientState(GL_VERTEX_ARRAY);
   //glDisableClientState(GL_TEXTURE_COORD_ARRAY);

   renderClusterBounds();

}

bool Q3BspRenderer :: checkCollision(const Vector3d *pos,Vector3d *pos_out) {
   
   Vector3d _pos_out(*pos);
   
   q3bsp_tBSPLeaf *currLeaf = findLeaf(pos);

   int i = currLeaf->leafface;
   int ii = currLeaf->numOfLeafFaces;
   while (ii--) {
   
      q3bsp_tBSPFace *face = &faces[leafFaces[i]];
      float *pnormal = face->vNormal;
      
      // all faces are built using triangle strips..
      // because of this fact, triangle count on any face is numOfFaceVertices - 2;
      int triCount = face->numOfVerts - 2;
      int v = face->startVertIndex;

      float planeD = - (pnormal[0] * vertices[v].vPosition[0] +
                         pnormal[1] * vertices[v].vPosition[1] +
                          pnormal[2] * vertices[v].vPosition[2]);
      

      // check all triangles for collision
      // FIXME: since all triangles lie on the same plane (they make a face!)
      // there is no need to check every face triangle for point classification
      // this must be optimized..
      while (triCount > 0) {

         bool collided = false;
      
         float *vertex[3];
         vertex[0] = vertices[v].vPosition;
         vertex[1] = vertices[v+1].vPosition;
         vertex[2] = vertices[v+2].vPosition;
      
         // classify point
         float my_distance = pnormal[0]*pos->x+pnormal[1]*pos->y+pnormal[2]*pos->z + planeD;
         
         if (abs(my_distance) <= 5.0f) {
            // plane intersection!!
         
            // calculate plane intersection point (pointOnPlane)
            Vector3d offset(my_distance*pnormal[0],my_distance*pnormal[1],my_distance*pnormal[2]);
            Vector3d pointOnPlane(*pos);
            pointOnPlane.sub(&offset);

            // now test if intersection point lies inside our triangle
            
            // find edge vectors
            Vector3d PV0(vertex[0][0]-pointOnPlane.x,
                          vertex[0][1]-pointOnPlane.y,
                           vertex[0][2]-pointOnPlane.z);
            Vector3d PV1(vertex[1][0]-pointOnPlane.x,
                          vertex[1][1]-pointOnPlane.y,
                           vertex[1][2]-pointOnPlane.z);
            Vector3d PV2(vertex[2][0]-pointOnPlane.x,
                          vertex[2][1]-pointOnPlane.y,
                           vertex[2][2]-pointOnPlane.z);

            PV0.normalize();
            PV1.normalize();
            PV2.normalize();

            // find inner angle sum
            float angle;
            angle = PV0.angle(&PV1);
            angle += PV1.angle(&PV2);
            angle += PV2.angle(&PV0);
            
            // if its >= a full circle than we are in triangle
            if (angle >= 360.0f * 0.99f) {
               collided = true;
            }

            // and check for polygon edge collision
            for(int i=0;i<3;++i) {
               
               int i_n = (i+1)%3;
               
               Vector3d closestPointOnLine;
               
               Vector3d v1(pointOnPlane.x-vertex[i][0],pointOnPlane.y-vertex[i][1],pointOnPlane.z-vertex[i][2]);
               Vector3d ndir(vertex[i_n][0]-vertex[i][0],vertex[i_n][1]-vertex[i][1],vertex[i_n][2]-vertex[i][2]);
               ndir.normalize();

               float dist = (float) sqrt(vertex[i][0]*vertex[i_n][0] +
                                          vertex[i][1]*vertex[i_n][1] +
                                           vertex[i][2]*vertex[i_n][2]);

               float proj_dist = ndir.dot(&v1);
               if (proj_dist <= 0)
                  closestPointOnLine.set(vertex[i][0],vertex[i][1],vertex[i][2]);
               else
               if (proj_dist >= dist)
                  closestPointOnLine.set(vertex[i_n][0],vertex[i_n][1],vertex[i_n][2]);
               else {
                  Vector3d vec(ndir.x*proj_dist,ndir.y*proj_dist,ndir.z*proj_dist);
                  closestPointOnLine.set(vertex[i][0]+vec.x,vertex[i][1]+vec.y,vertex[i][2]+vec.z);
               }


               float distance = (float) sqrt(pointOnPlane.x*closestPointOnLine.x+
                                              pointOnPlane.y*closestPointOnLine.y+
                                               pointOnPlane.z*closestPointOnLine.z);
               if (distance < 0.5f) {
                  collided = true;
                  break;
               }
    
            }
         }

         if (collided) {

            Vector3d colOffset(0.0f,0.0f,0.0f);

            if(my_distance > 0) {
               float distanceOver = 1.0f - my_distance;
               colOffset.set(pnormal[0]*distanceOver,pnormal[1]*distanceOver,pnormal[2]*distanceOver);
            }
            else {
               float distanceOver = 1.0f + my_distance;
               colOffset.set(-pnormal[0]*distanceOver,-pnormal[1]*distanceOver,-pnormal[2]*distanceOver);
            }

            _pos_out.add(&colOffset);

         }
                  
         v++;
         triCount--;
      }
   
   
      // next leaf face
      i++;
   }
   
   return false;
}

void Q3BspRenderer :: renderClusterBounds() {

   for (int i=0;i<leafCount;++i) {
      q3bsp_tBSPLeaf *leaf = &leafs[i];
   }

}

