#include <lib3d/app/BaseApp.h>

BaseApp :: BaseApp() {
   quit_app = false;
   video_initialized = false;
   
   simpleCamera.pos = Vector3d(0,0,0);
   simpleCamera.xangle = 0.0f;
   simpleCamera.yangle = 0.0f;
   simpleCamera.zangle = 0.0f;
   simple_cam_forw = false;
   simple_cam_back = false;
   simple_cam_tleft = false;
   simple_cam_tright = false;
   simple_cam_up = false;
   simple_cam_down = false;
   simple_cam_tup = false;
   simple_cam_tdown = false;
   setSimpleCameraSpeed(1.0f);
   enableSimpleCamera(false);

   videoFlags = 0;
   framesPerSecond = 0;
   frameCount = 0;
   
   setSize(640,480);
   setColorBits(32);
   setVideoFlags(SDL_OPENGL);
   setTitle("lib3d window");
   showFPS(false);
   setPerspectiveParams(45.0f,0.1f,10000.0f);
   setOrthoParams(0.0f,1.0f,0.0f,1.0f);
   setProjection(PROJ_PERSPECTIVE);
   
}

void BaseApp :: setSize(int w,int h) {
   width = w;
   height = h;
}

void BaseApp :: setTitle(const char *title) {
  if (title)
      strcpy(windowTitle,title);
   else
      strcpy(windowTitle," ");

   if (video_initialized)
      SDL_WM_SetCaption(windowTitle,NULL);
}

void BaseApp :: setColorBits(int bpp) {
   this->bpp = bpp;
}

void BaseApp :: setFullScreen(bool enable) {
   if (enable)
      setVideoFlags(SDL_FULLSCREEN);
   else
      videoFlags &= ~SDL_FULLSCREEN;
}

void BaseApp :: showFPS(bool enable) {
   show_fps = enable;
}

void BaseApp :: setPerspectiveParams(float vfov,float neard,float fard) {
   fov = vfov;
   this->neard = neard;
   this->fard = fard;
   onWindowResize(width,height);
}

void BaseApp :: setOrthoParams(float left,float right,float bottom,float top) {
   this->left = left;
   this->right = right;
   this->top = top;
   this->bottom = bottom;
   onWindowResize(width,height);
}

void BaseApp :: setProjection(enum eProjectionType proj) {
   projType = proj;
   onWindowResize(width,height);
}

void BaseApp :: setVideoFlags(int flagbits) {
   videoFlags |= flagbits;
}

unsigned int BaseApp :: getFPS() {
   return framesPerSecond;
}

void BaseApp :: enableSimpleCamera(bool enable) {
   use_simple_camera = enable;
}

void BaseApp :: setSimpleCameraSpeed(float spd) {
   simple_camera_speed = spd;
}

void BaseApp :: onWindowResize(int w,int h) {

   if (h == 0)
      h = 1;

   setSize(w,h);
   
   glViewport(0,0,width,height);
   glMatrixMode(GL_PROJECTION);
   glLoadIdentity();
   
   if (projType == PROJ_PERSPECTIVE)
      gluPerspective(fov,(float)width/(float)height,neard,fard);
   else
   if (projType == PROJ_ORTHO)
      gluOrtho2D(left,right,bottom,top);
      
   glMatrixMode(GL_MODELVIEW);

}

void BaseApp :: start() {
   
   initVideo();
   
#ifdef WIN32
   initLib3d(true);
#else
   initLib3d(false);
#endif

   initOpenGL();
   onInit();
   run();
}

void BaseApp :: exit() {
   onExit();
   quit_app = true;
}

void BaseApp :: updateSimpleCamera(const SDLKey &key,bool pressed) {

   switch (key) {
   
      case SDLK_UP:
         if (pressed) {
            simple_cam_forw = true;
            simple_cam_back = false;
         }
         else
            simple_cam_forw = false;
         break;
   
      case SDLK_DOWN:
         if (pressed) {
            simple_cam_back = true;
            simple_cam_forw = false;
         }
         else
            simple_cam_back = false;
         break;
   
      case SDLK_LEFT:
         if (pressed) {
            simple_cam_tleft = true;
            simple_cam_tright = false;
         }
         else
            simple_cam_tleft = false;
         break;
   
      case SDLK_RIGHT:
         if (pressed) {
            simple_cam_tright = true;
            simple_cam_tleft = false;
         }
         else
            simple_cam_tright = false;
         break;
   
      case SDLK_PAGEUP:
         if (pressed) {
            simple_cam_up = true;
            simple_cam_down = false;
         }
         else
            simple_cam_up = false;
         break;
   
      case SDLK_PAGEDOWN:
         if (pressed) {
            simple_cam_down = true;
            simple_cam_up = false;
         }
         else
            simple_cam_down = false;
         break;

      case SDLK_w:
         if (pressed) {
            simple_cam_tup = true;
            simple_cam_tdown = false;
         }
         else
            simple_cam_tup = false;
         break;
         
      case SDLK_s:
         if (pressed) {
            simple_cam_tdown = true;
            simple_cam_tup = false;
         }
         else
            simple_cam_tdown = false;
         break;
         
   };

}

void BaseApp :: moveSimpleCamera() {

   if (simple_cam_forw)
      simpleCamera.moveForward(simple_camera_speed);
   else
   if (simple_cam_back)
      simpleCamera.moveBackward(simple_camera_speed);

   if (simple_cam_tleft)
      simpleCamera.turnLeft(simple_camera_speed);
   else
   if (simple_cam_tright)
      simpleCamera.turnRight(simple_camera_speed);

   if (simple_cam_up)
      simpleCamera.moveUp(simple_camera_speed);
   else
   if (simple_cam_down)
      simpleCamera.moveDown(simple_camera_speed);

   if (simple_cam_tup)
      simpleCamera.turnUp(simple_camera_speed);
   else
   if (simple_cam_tdown)
      simpleCamera.turnDown(simple_camera_speed);

}

void BaseApp :: applySimpleCamera() {

   glRotatef(simpleCamera.xangle,1,0,0);
   glRotatef(360.0f-simpleCamera.yangle,0,1,0);
   //glRotatef(360.0f-simpleCamera.zangle,0,0,1);
   glTranslatef(-simpleCamera.pos.x,-simpleCamera.pos.y,-simpleCamera.pos.z);

}

void BaseApp :: initVideo() {

   int  errNum=0;
      
   if (SDL_Init(SDL_INIT_VIDEO | SDL_INIT_TIMER) < 0) {
      errNum = 1;      
   }
	
   if (SDL_SetVideoMode(width,height,bpp,videoFlags) == NULL) {
      errNum = 2;
   }

   SDL_WM_SetCaption(windowTitle,NULL);

   if (errNum != 0) {
      FILE *errStream = fopen("BaseApp_ERROR.txt","wt");
      
      switch (errNum) {
         case 1:
            fprintf(errStream,"SDL baslatiminda hata! %s\n",SDL_GetError());
            break;
         case 2:
            fprintf(errStream,"SDL pencereyi acamadi!\n");
            break;
         default:
            fprintf(errStream,"tanimlanamayan hata!!!\n");
      };
      
      fclose(errStream);
   }

   video_initialized = true;

}

void BaseApp :: initOpenGL() {

   glDepthFunc(GL_LESS);
   glEnable(GL_DEPTH_TEST);
   glShadeModel(GL_SMOOTH);
   glEnable(GL_NORMALIZE);
   glEnable(GL_CULL_FACE);
   glFrontFace(GL_CCW);
   glClearColor(0,0,0,0);
   glDisable(GL_TEXTURE_2D);
   glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);

   glEnable(GL_LIGHTING);
   glEnable(GL_LIGHT0);
   glEnable(GL_COLOR_MATERIAL);
   
   onWindowResize(width,height);
}

void BaseApp :: run() {
   
   lastTime = SDL_GetTicks();
   
   do {

      if (use_simple_camera) {
         moveSimpleCamera();
      }
      
      onRender();
      SDL_GL_SwapBuffers();

      ++frameCount;
      unsigned int now = SDL_GetTicks();
      if (now - lastTime >= 1000) {
         framesPerSecond = frameCount;
         frameCount = 0;
         lastTime = now;

         if (show_fps) {
            static char fps[256];
            sprintf(fps,"FPS: %d",framesPerSecond);
            setTitle(fps);
         }


      }

      SDL_Event event;
      while (SDL_PollEvent(&event)) {
         
         switch (event.type) {
         
            case SDL_KEYDOWN:
               onKeyPressed(event.key.keysym.sym);
               if (use_simple_camera)
                  updateSimpleCamera(event.key.keysym.sym,true);
               break;

            case SDL_KEYUP:
               onKeyReleased(event.key.keysym.sym);
               if (use_simple_camera)
                  updateSimpleCamera(event.key.keysym.sym,false);
               break;         

            case SDL_MOUSEBUTTONDOWN:
               onMouseButtonPressed(event.button.button,event.button.x,event.button.y);
               break;

            case SDL_MOUSEBUTTONUP:
               onMouseButtonReleased(event.button.button,event.button.x,event.button.y);
               break;
            
            case SDL_VIDEORESIZE:
               onWindowResize(event.resize.w,event.resize.h);
               break;

            case SDL_QUIT:
               exit();
               break;         
                     
            default:
               onEvent(event);
         }

      }


   } while (!quit_app);

   SDL_Quit();

}



