{
  OyunYapimi.org DOT3 Per-Pixel Lighting
    Programlayan  : Osman Turan
    Son Gncelleme: 22.08.2004

  ablon Kod Bilgileri:
    Programlayan: Maarten Kronberger
    Website     : http://www.sulaco.co.za
}

program OpenGLApp;

uses
  Windows,
  Messages,
  dglOpenGL,
  Textures,
  Math;

type
// doku koordinat tipimizi belirleyelim
  TGLArrayf2 = array[0..1] of TGLFloat;
// 3x3 matris belirleyelim
  TGLMatrixf3 = array[0..3] of TGLArrayf3;

// alma zamani vertex tipimizi belirleyelim
  TVertex = packed record
    XYZ: TGLArrayf3;  // pozisyon
    Binormal: TGLArrayf3;  // binormal vektr
    Tangent: TGLArrayf3;   // tangent vektr
    Normal: TGLArrayf3;    // normal vektr
    ST: TGLArrayf2;  // doku koordinat
  end;

const
  WND_TITLE = 'OyunYapimi.org Per-Pixel Lighting Tutorial by Osman Turan';
  FPS_TIMER = 1;                     // Timer to calculate FPS
  FPS_INTERVAL = 1000;               // Calculate FPS every 1000 ms

// Modelin indekslerini tanmlayalm (GL_TRIANGLES'a uygun)
  ModelElements: array[0..35] of Cardinal = (
     0,  1,  2,  2,  3,  0, // n yz
     4,  5,  6,  6,  7,  4, // arka yz
     8,  9, 10, 10, 11,  8, // st yz
    12, 13, 14, 14, 15, 12, // alt yz
    16, 17, 18, 18, 19, 16, // sa yz
    20, 21, 22, 22, 23, 20  // sol yz
  );

// "Per-Pixel Lighting" vektorlerini hesaplayacak vertex programmz
// vertex.texcoord[0] -> Ana doku koordinatlar
// vertex.normal      -> Normal vektr
// vertex.color
//   Giri            -> Binormal vektr
//   k            -> Detay Doku tarafndan kullanlacak k vektorleri
// vertex.texcoord[1] -> Tangent vektr
  VP_PerPixelLighting: String =
{1} '!!ARBvp1.0'+#10+                           // ARB vertex program 1.0 bal
{2} 'OPTION ARB_position_invariant;'+#10+       // vertexleri homojen koordinatlara evir

{3} 'PARAM light = program.local[0];'+#10+      // "light" k koordinatini tutuyor
    'PARAM half = { 0.5, 0.5, 0.5, 0.5 };'+#10+ // 0.5 katsays

{4} 'ATTRIB normal = vertex.normal;'+#10+       // giri normal
{5} 'ATTRIB binormal = vertex.color;'+#10+      // giri renk
{6} 'ATTRIB tangent = vertex.texcoord[1];'+#10+ // giri doku koordinat (1)

{7} 'OUTPUT tex0 = result.texcoord[0];'+#10+    // sonu doku koordinat (0)
{8} 'OUTPUT tex1 = result.texcoord[1];'+#10+    // sonu doku koordinat (1)
{9} 'OUTPUT lightvec = result.color;'+#10+      // sonu renk (k vektrleri)
{10}'TEMP vec, R0, R1, R2, R3, R4, R5;'+#10+    // program deiken tanmlanmas

{11}'ADD vec, light, -vertex.position;'+#10+    // k vektrn hesapla
{12}'DP3 R0.x, vec, tangent;'+#10+              // k vektrn doku dzlemine indirge
{13}'DP3 R0.y, vec, binormal;'+#10+
{14}'DP3 R0.z, vec, normal;'+#10+

{15}'DP3 R1, R0, R0;'+#10+                      // k vektrne normalizasyon uygula
{16}'RSQ R2, R1.x;'+#10+
{17}'MUL R3, R0, R2;'+#10+
{18}'MUL R4, R3, half.yyyy;'+#10+               // [0, 1] aralna indirge (DOT3 ilemi iin)
{19}'ADD R5, R4, half.yyyy;'+#10+
{20}'MOV lightvec, R5;'+#10+                    // birincil renkte sakla (primary color)

{21}'MOV tex0, vertex.texcoord[0];'+#10+        // ana doku koordinatlarini tekrar gnder
{22}'MOV tex1, vertex.texcoord[0];'+#10+        // detay doku ana dokunun koordinatlarini gostersin
                                                // aksi taktirde koordinatlar geersiz olur
{23}'END';                                      // son

var
  h_Wnd  : HWND;                     // Global window handle
  h_DC   : HDC;                      // Global device context
  h_RC   : HGLRC;                    // OpenGL rendering context
  keys : Array[0..255] of Boolean;   // Holds keystrokes
  FPSCount : Integer = 0;            // Counter for FPS
  ElapsedTime : Integer;             // Elapsed time between frames

  // Textures
  Decalmap: GLuint;                  // ana doku
  Normalmap: GLuint;                 // detay doku

  // User vaiables
  VP_DOT3_Handle: GLuint;            // vertex program tutac
  DeltaTime: Double;                 // son iki izim karesi arasnda geen zaman
  xSpeed, ySpeed : glFloat;          // dnme hz
  xAngle, yAngle : glFloat;          // dnme as

  BumpActive: Boolean;               // bumpmapping etkin/etkisiz

  LightOrigin: TGLArrayf3 = (1.5, 1, 1.75); // Ik kaynann koordinatlar

// Kp modelimizi kelerini tanmlayalm (tanjant uzay sonra hesaplayacaz)
  ModelVertices: array[0..23] of TVertex = (
  // n yz
    (XYZ: (-1, -1, +1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 0,  0,  1); ST: (0, 0)),
    (XYZ: (+1, -1, +1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 0,  0,  1); ST: (0, 1)),
    (XYZ: (+1, +1, +1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 0,  0,  1); ST: (1, 1)),
    (XYZ: (-1, +1, +1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 0,  0,  1); ST: (1, 0)),
  // arka yz
    (XYZ: (-1, -1, -1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 0,  0, -1); ST: (0, 1)),
    (XYZ: (-1, +1, -1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 0,  0, -1); ST: (1, 1)),
    (XYZ: (+1, +1, -1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 0,  0, -1); ST: (1, 0)),
    (XYZ: (+1, -1, -1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 0,  0, -1); ST: (0, 0)),
  // st yz
    (XYZ: (-1, +1, -1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 0,  1,  0); ST: (1, 0)),
    (XYZ: (-1, +1, +1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 0,  1,  0); ST: (0, 0)),
    (XYZ: (+1, +1, +1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 0,  1,  0); ST: (0, 1)),
    (XYZ: (+1, +1, -1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 0,  1,  0); ST: (1, 1)),
  // alt yz
    (XYZ: (-1, -1, -1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 0, -1,  0); ST: (1, 1)),
    (XYZ: (+1, -1, -1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 0, -1,  0); ST: (1, 0)),
    (XYZ: (+1, -1, +1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 0, -1,  0); ST: (0, 0)),
    (XYZ: (-1, -1, +1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 0, -1,  0); ST: (0, 1)),
  // sa yz
    (XYZ: (+1, -1, -1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 1,  0,  0); ST: (0, 1)),
    (XYZ: (+1, +1, -1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 1,  0,  0); ST: (1, 1)),
    (XYZ: (+1, +1, +1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 1,  0,  0); ST: (1, 0)),
    (XYZ: (+1, -1, +1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 1,  0,  0); ST: (0, 0)),
  // sol yz
    (XYZ: (-1, -1, -1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: (-1,  0,  0); ST: (0, 0)),
    (XYZ: (-1, -1, +1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: (-1,  0,  0); ST: (0, 1)),
    (XYZ: (-1, +1, +1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: (-1,  0,  0); ST: (1, 1)),
    (XYZ: (-1, +1, -1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: (-1,  0,  0); ST: (1, 0))
{    (XYZ: (-1, -1, +1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 0,  0,  1); ST: (0, 0)),
    (XYZ: (+1, -1, +1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 0,  0,  1); ST: (1, 0)),
    (XYZ: (+1, +1, +1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 0,  0,  1); ST: (1, 1)),
    (XYZ: (-1, +1, +1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 0,  0,  1); ST: (0, 1)),
  // arka yz
    (XYZ: (-1, -1, -1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 0,  0, -1); ST: (1, 0)),
    (XYZ: (-1, +1, -1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 0,  0, -1); ST: (1, 1)),
    (XYZ: (+1, +1, -1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 0,  0, -1); ST: (0, 1)),
    (XYZ: (+1, -1, -1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 0,  0, -1); ST: (0, 0)),
  // st yz
    (XYZ: (-1, +1, -1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 0,  1,  0); ST: (0, 1)),
    (XYZ: (-1, +1, +1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 0,  1,  0); ST: (0, 0)),
    (XYZ: (+1, +1, +1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 0,  1,  0); ST: (1, 0)),
    (XYZ: (+1, +1, -1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 0,  1,  0); ST: (1, 1)),
  // alt yz
    (XYZ: (-1, -1, -1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 0, -1,  0); ST: (1, 1)),
    (XYZ: (+1, -1, -1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 0, -1,  0); ST: (0, 1)),
    (XYZ: (+1, -1, +1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 0, -1,  0); ST: (0, 0)),
    (XYZ: (-1, -1, +1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 0, -1,  0); ST: (1, 0)),
  // sa yz
    (XYZ: (+1, -1, -1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 1,  0,  0); ST: (1, 0)),
    (XYZ: (+1, +1, -1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 1,  0,  0); ST: (1, 1)),
    (XYZ: (+1, +1, +1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 1,  0,  0); ST: (0, 1)),
    (XYZ: (+1, -1, +1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: ( 1,  0,  0); ST: (0, 0)),
  // sol yz
    (XYZ: (-1, -1, -1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: (-1,  0,  0); ST: (0, 0)),
    (XYZ: (-1, -1, +1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: (-1,  0,  0); ST: (1, 0)),
    (XYZ: (-1, +1, +1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: (-1,  0,  0); ST: (1, 1)),
    (XYZ: (-1, +1, -1); Binormal: (0, 0, 0); Tangent: (0, 0, 0); Normal: (-1,  0,  0); ST: (0, 1))}
  );  

  // Mouse Variables
  MouseMove : boolean = false;        // Is the mouse moving?
  MoveXcoord, MoveYcoord : GLfloat;   // Current X and Y position of the cursor

  MouseButton : Integer = -1;         // mouse button down
  Xcoord, Ycoord : Integer;           // Current X and Y position where the mouse was clicked

  OpenGLInitialized : Boolean;

{$R *.RES}

procedure glKillWnd(Fullscreen : Boolean); forward;

{------------------------------------------------------------------}
{  Function to convert int to string. (No sysutils = smaller EXE)  }
{------------------------------------------------------------------}
function IntToStr(Num : Integer) : String;  // using SysUtils increase file size by 100K
begin
  Str(Num, result);
end;

{------------------------------------------------------------------}
{  Baz vektrel matematik ilemleri                               }
{------------------------------------------------------------------}
function VectorRotateX(v : TGLArrayf3; angle : Single):TGLArrayf3;
Var s,c: Extended;
Begin
  SinCos(Angle, s, c);
  Result[0] := v[0];
  Result[1] := v[1] * c + v[2] * -s;
  Result[2] := v[1] * s + v[2] * c;
End;

function VectorRotateY(v : TGLArrayf3; angle : Single):TGLArrayf3;
Var s,c : Extended;
Begin
  SinCos(Angle, s, c);
  Result[0] := v[0] * c + v[2] * s;
  Result[1] := v[1];
  Result[2] := v[0] * -s + v[2] * c;
End;

function VectorSubtract(v1, v2: TGLArrayf3): TGLArrayf3;
begin
  Result[0] := v1[0] - v2[0];
  Result[1] := v1[1] - v2[1];
  Result[2] := v1[2] - v2[2];
end;

function VectorNegate(v: TGLArrayf3): TGLArrayf3;
begin
  Result[0] := -v[0];
  Result[1] := -v[1];
  Result[2] := -v[2];
end;

function VectorDotProduct(v1, v2: TGLArrayf3): TGLFloat;
begin
  Result := (v1[0] * v2[0]) +
            (v1[1] * v2[1]) +
            (v1[2] * v2[2]);
end;

function VectorNormalize(v: TGLArrayf3): TGLArrayf3;
var len: Single;
begin
  len := Sqrt((v[0] * v[0]) +
              (v[1] * v[1]) +
              (v[2] * v[2]));

  if len <= 0 then
  begin
    Result[0] := 0;
    Result[1] := 0;
    Result[2] := 0;
  end
  else
  begin
    len := 1 / len;
    Result[0] := v[0] * len;
    Result[1] := v[1] * len;
    Result[2] := v[2] * len;
  end;
end;

function VectorLerp(v1, v2: TGLArrayf3; t: Single): TGLArrayf3;
begin
  Result[0] := v1[0] + (v2[0] - v1[0]) * t;
  Result[1] := v1[1] + (v2[1] - v1[1]) * t;
  Result[2] := v1[2] + (v2[2] - v1[2]) * t;
end;

function VectorTransform(V: TGLArrayf3; M: TMatrix4f): TGLArrayf3;
begin
  Result[0]:=V[0] * M[0, 0] + V[1] * M[1, 0] + V[2] * M[2, 0] + M[3, 0];
  Result[1]:=V[0] * M[0, 1] + V[1] * M[1, 1] + V[2] * M[2, 1] + M[3, 1];
  Result[2]:=V[0] * M[0, 2] + V[1] * M[1, 2] + V[2] * M[2, 2] + M[3, 2];
end;

procedure TransposeMatrix(var M: TMatrix4f);
var
   f : Single;
begin
   f:=M[0, 1]; M[0, 1]:=M[1, 0]; M[1, 0]:=f;
   f:=M[0, 2]; M[0, 2]:=M[2, 0]; M[2, 0]:=f;
   f:=M[0, 3]; M[0, 3]:=M[3, 0]; M[3, 0]:=f;
   f:=M[1, 2]; M[1, 2]:=M[2, 1]; M[2, 1]:=f;
   f:=M[1, 3]; M[1, 3]:=M[3, 1]; M[3, 1]:=f;
   f:=M[2, 3]; M[2, 3]:=M[3, 2]; M[3, 2]:=f;
end;

{------------------------------------------------------------------}
{  Bir assembly vertex programn OpenGL srcsne ykler.        }
{------------------------------------------------------------------}
procedure LoadVertexProgram(VPText : String; var VPHandle : cardinal);
var
  errPos : Integer;
  errString : String;
begin
  glGenProgramsARB(1, @VPHandle);
  glBindProgramARB(GL_VERTEX_PROGRAM_ARB, VPHandle);
  glProgramStringARB(GL_VERTEX_PROGRAM_ARB,GL_PROGRAM_FORMAT_ASCII_ARB,
                     Length(VPText), PChar(VPText));
  glGetIntegerv(GL_PROGRAM_ERROR_POSITION_ARB, @errPos);
  if errPos>-1 then begin
    errString:=glGetString(GL_PROGRAM_ERROR_STRING_ARB);
    MessageBox(h_Wnd, PChar('Vertex programda hata tespit edildi: '+ #13#10 + errString),
               nil, MB_OK or MB_ICONERROR);
    glKillWnd(False);
    Halt;
  end;
end;

{-------------------------------------------------------------------}
{ Doku koordinatlar ve normal vektrlerinden yola karak tanjant  }
{ uzay vektrlerini dzenler (GLScene'den adapte edilmitir)        }
{ GLScene - http://www.glscene.org/                                 }
{-------------------------------------------------------------------}
procedure SetupTangentSpace(var Vertices: array of TVertex;
                            const Elements: array of Cardinal;
                            const VertexCount: Integer;
                            const ElemCount: Integer);
var
  i,j        : Integer;
  v,n,t      : TGLMatrixf3;
  vt,tt      : TGLArrayf3;
  interp,dot : Single;

  procedure SortVertexData(sortidx : Integer);
  begin
    if t[0][sortidx]<t[1][sortidx] then begin
      vt:=v[0];   tt:=t[0];
      v[0]:=v[1]; t[0]:=t[1];
      v[1]:=vt;   t[1]:=tt;
    end;
    if t[0][sortidx]<t[2][sortidx] then begin
      vt:=v[0];   tt:=t[0];
      v[0]:=v[2]; t[0]:=t[2];
      v[2]:=vt;   t[2]:=tt;
    end;
    if t[1][sortidx]<t[2][sortidx] then begin
      vt:=v[1];   tt:=t[1];
      v[1]:=v[2]; t[1]:=t[2];
      v[2]:=vt;   t[2]:=tt;
    end;
  end;

begin
  for i:=0 to (ElemCount div 3)-1 do begin
    // Get triangle data
    for j:=0 to 2 do begin
      v[j]:=Vertices[Elements[3*i+j]].XYZ;
      n[j]:=Vertices[Elements[3*i+j]].Normal;
      t[j,0]:=Vertices[Elements[3*i+j]].ST[0];
      t[j,1]:=Vertices[Elements[3*i+j]].ST[1];
    end;

    for j:=0 to 2 do begin
      // Compute tangent
      SortVertexData(1);

      if (t[2][1]-t[0][1]) = 0 then interp:=1
      else interp:=(t[1][1]-t[0][1])/(t[2][1]-t[0][1]);

      vt:=VectorLerp(v[0],v[2],interp);
      interp:=t[0][0]+(t[2][0]-t[0][0])*interp;
      vt:=VectorSubtract(vt,v[1]);
      if t[1][0]<interp then vt:=VectorNegate(vt);
      dot:=VectorDotProduct(vt,n[j]);
      vt[0]:=vt[0]-n[j][0]*dot;
      vt[1]:=vt[1]-n[j][1]*dot;
      vt[2]:=vt[2]-n[j][2]*dot;
      Vertices[Elements[3*i+j]].Tangent := VectorNormalize(vt);

      // Compute Bi-Normal
      SortVertexData(0);

      if (t[2][0]-t[0][0]) = 0 then interp:=1
      else interp:=(t[1][0]-t[0][0])/(t[2][0]-t[0][0]);

      vt:=VectorLerp(v[0],v[2],interp);
      interp:=t[0][1]+(t[2][1]-t[0][1])*interp;
      vt:=VectorSubtract(vt,v[1]);
      if t[1][1]<interp then vt:=VectorNegate(vt);
      dot:=VectorDotProduct(vt,n[j]);
      vt[0]:=vt[0]-n[j][0]*dot;
      vt[1]:=vt[1]-n[j][1]*dot;
      vt[2]:=vt[2]-n[j][2]*dot;
      Vertices[Elements[3*i+j]].Binormal := VectorNormalize(vt);
    end;
  end;
end;

{------------------------------------------------------------------}
{  Function to draw the actual scene                               }
{------------------------------------------------------------------}
procedure glDraw();
var
  matrix: TMatrix4f;
  light: TGLArrayf3;
  param: TGLArrayf4;
begin
// gereken tamponlar sil
  glClear(GL_COLOR_BUFFER_BIT or GL_DEPTH_BUFFER_BIT);

// sahneyi biraz dzenle
  glLoadIdentity();

  glTranslatef(0.0,0.0,-5);

  glRotatef(xAngle, 1, 0, 0);
  glRotatef(yAngle, 0, 1, 0);

// modelview matrix ters evirip k kaynan gerek koordinatlarna
// yerletir. Aksi taktirde vertex program yanl hesap yapar.
  glGetFloatv(GL_MODELVIEW_MATRIX, @matrix[0]);
  TransposeMatrix(matrix);
  light := VectorTransform(LightOrigin, matrix);
// vertex programn ileyebilecei 4 float lk yapya dntr
  param[0] := light[0];
  param[1] := light[1];
  param[2] := light[2];
  param[3] := 1;

// vertex program ve tm doku birimlerini etkisiz yap
  glDisable(GL_VERTEX_PROGRAM_ARB);
  glActiveTextureARB(GL_TEXTURE0_ARB);
  glClientActiveTextureARB(GL_TEXTURE0_ARB);
  glDisable(GL_TEXTURE_2D);

  glActiveTextureARB(GL_TEXTURE1_ARB);
  glClientActiveTextureARB(GL_TEXTURE1_ARB);
  glDisable(GL_TEXTURE_2D);

// Ik koordinatnda 5 piksellik bir nokta izdir
  glPointSize(5);
  glBegin(GL_POINTS);
  glVertex3fv(@light[0]);
  glEnd;

  if BumpActive then
  begin
  // vertex program etkin yap
    glEnable(GL_VERTEX_PROGRAM_ARB);
    glBindProgramARB(GL_VERTEX_PROGRAM_ARB, VP_DOT3_Handle);
    glProgramLocalParameter4fvARB(GL_VERTEX_PROGRAM_ARB, 0, @param[0]);
  end;

// koordinatlari OpenGL srcsne gnder
  glEnableClientState(GL_VERTEX_ARRAY);
  glVertexPointer(3, GL_FLOAT, SizeOf(TVertex), @ModelVertices[0].XYZ[0]);
  glEnableClientState(GL_NORMAL_ARRAY);
  glNormalPointer(GL_FLOAT, SizeOf(TVertex), @ModelVertices[0].Normal[0]);

  if BumpActive then
  begin
    glEnableClientState(GL_COLOR_ARRAY);
    glColorPointer(3, GL_FLOAT, SizeOf(TVertex), @ModelVertices[0].Binormal[0]);
  end;

// birinci (varsaylan) doku birimini aktif yap
  glActiveTextureARB(GL_TEXTURE0_ARB);
  glClientActiveTextureARB(GL_TEXTURE0_ARB);
  glEnable(GL_TEXTURE_2D);

// detay dokuyu aktif yap
  if BumpActive then
    glBindTexture(GL_TEXTURE_2D, Normalmap);

// koordinatlari srcye gnder (hepsi gerekmiyor)
  glEnableClientState(GL_TEXTURE_COORD_ARRAY);
  glTexCoordPointer(2, GL_FLOAT, SizeOf(TVertex), @ModelVertices[0].ST[0]);

  if BumpActive then
  begin
  // DOT3 texture combine moduna ge
    glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_COMBINE_ARB);
    glTexEnvi(GL_TEXTURE_ENV, GL_COMBINE_RGB_ARB, GL_DOT3_RGB_ARB);
  // ilk parametre birincil renk (primary color) yani k vektrleri
    glTexEnvi(GL_TEXTURE_ENV, GL_SOURCE0_RGB_ARB, GL_PRIMARY_COLOR_ARB);
    glTexEnvi(GL_TEXTURE_ENV, GL_OPERAND0_RGB_ARB, GL_SRC_COLOR);
  // ikinci parametre detay dokusu (normalmap)
    glTexEnvi(GL_TEXTURE_ENV, GL_SOURCE1_RGB_ARB, GL_TEXTURE);
    glTexEnvi(GL_TEXTURE_ENV, GL_OPERAND1_RGB_ARB, GL_SRC_COLOR);
  end
  else // modulate combine moduna geri evir
    glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

  if BumpActive then
  begin
  // ikinci doku birimini aktif yap
    glActiveTextureARB(GL_TEXTURE1_ARB);
    glClientActiveTextureARB(GL_TEXTURE1_ARB);
    glEnable(GL_TEXTURE_2D);
  end;

// ana dokuyu aktif yap
  glBindTexture(GL_TEXTURE_2D, Decalmap);

  if BumpActive then
  begin
  // tanjant vektrlerini ikincil doku kanal zerinden doku koordinat olarak gnder
    glEnableClientState(GL_TEXTURE_COORD_ARRAY);
    glTexCoordPointer(2, GL_FLOAT, SizeOf(TVertex), @ModelVertices[0].Tangent[0]);

  // ikincil doku birimi iin MODULATE ilemini hazirla
    glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_COMBINE_ARB);
    glTexEnvi(GL_TEXTURE_ENV, GL_COMBINE_RGB_ARB, GL_MODULATE);
  // ilk parametre nceki doku birimi
    glTexEnvi(GL_TEXTURE_ENV, GL_SOURCE0_RGB_ARB, GL_PREVIOUS_ARB);
    glTexEnvi(GL_TEXTURE_ENV, GL_OPERAND0_RGB_ARB, GL_SRC_COLOR);
  // ikinci parametre ana doku
    glTexEnvi(GL_TEXTURE_ENV, GL_SOURCE1_RGB_ARB, GL_TEXTURE);
    glTexEnvi(GL_TEXTURE_ENV, GL_OPERAND1_RGB_ARB, GL_SRC_COLOR);
  end;

// modeli izdir
  glDrawElements(GL_TRIANGLES, 36, GL_UNSIGNED_INT, @ModelElements[0]);

// renk kanaln kapat (daha sonra nokta izeceiz)
  glDisableClientState(GL_COLOR_ARRAY);

// Hza gre dndrme uygula
  xAngle :=xAngle + xSpeed;
  yAngle :=yAngle + ySpeed;

// bence burda bu komut gereksiz :)
// Flush the OpenGL Buffer
//  glFlush();                            // ( Force the buffer to draw or send a network packet of commands in a networked system)
end;


{------------------------------------------------------------------}
{  Initialise OpenGL                                               }
{------------------------------------------------------------------}
procedure glInit();
begin
// OpenGL src eklentilerimizi kontrol et
  if not (GL_ARB_multitexture) then
  begin
    MessageBox(0, 'GL_ARB_multitexture bulunamad', nil, MB_OK or MB_ICONERROR);
    glKillWnd(False);
    Halt;
  end;

  if not (GL_ARB_texture_env_combine) then
  begin
    MessageBox(0, 'GL_ARB_texture_env_combine bulunamad', nil, MB_OK or MB_ICONERROR);
    glKillWnd(False);
    Halt;
  end;

  if not (GL_ARB_texture_env_dot3) then
  begin
    MessageBox(0, 'GL_ARB_texture_env_dot3 bulunamad', nil, MB_OK or MB_ICONERROR);
    glKillWnd(False);
    Halt;
  end;

  if not (GL_ARB_vertex_program) then
  begin
    MessageBox(0, 'GL_ARB_vertex_program bulunamad', nil, MB_OK or MB_ICONERROR);
    glKillWnd(False);
    Halt;
  end;

// OpenGL varsaylan ayarlarn yap   
  glClearColor(0.0, 0.0, 0.0, 0.0); 	   // Black Background
  glShadeModel(GL_SMOOTH);                 // Enables Smooth Color Shading
  glClearDepth(1.0);                       // Depth Buffer Setup
  glEnable(GL_DEPTH_TEST);                 // Enable Depth Buffer
  glDepthFunc(GL_LESS);		           // The Type Of Depth Test To Do

  glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);   //Realy Nice perspective calculations
  glEnable(GL_CULL_FACE);

  glEnable(GL_TEXTURE_2D);                          // Enable Texture Mapping
  LoadTexture('decalmap.tga', Decalmap, True);     // ana dokuyu ykle

// Normalmap kltld zaman ierdii bilgi bozulur.
// nk, normalmap aslnda her bir pikselde kodlanm birim vektrlerden
// oluur. Eer sradan kltme ilemi uygularsak vektrler bozulur. nk
// sradan klme ilemleri bir grup pikselin ortalama deerlerini alarak
// yeni bir piksel oluturur. Buradaki hassasiyetten tr bozulmalar kanlmaz
// hale gelir. Bu sorundan kurtulmann iki yolu vardr:
//  1 - Vektr bazl kltme
//  2 - Hi kltme uygulamama (No Mipmapping)
// Biz burada ikinci seenei kullanacaz. Ama doku bykse birinci seenei
// kullanmakta yarar var.

  LoadTexture('normalmap.tga', Normalmap, False);   // detay dokusunu ykle

// vertex program ykle
  LoadVertexProgram(VP_PerPixelLighting, VP_DOT3_Handle);

// tanjant uzay hesapla
  SetupTangentSpace(ModelVertices, ModelElements, 24, 36);

// bumpmapping i etkin yap (varsaylan)
  BumpActive := True;

  xSpeed :=0.1;   // start with some movement
  ySpeed :=0.1;
end;


{------------------------------------------------------------------}
{  Handle window resize                                            }
{------------------------------------------------------------------}
procedure glResizeWnd(Width, Height : Integer);
begin
  if (Height = 0) then                // prevent divide by zero exception
    Height := 1;
  glViewport(0, 0, Width, Height);    // Set the viewport for the OpenGL window
  glMatrixMode(GL_PROJECTION);        // Change Matrix Mode to Projection
  glLoadIdentity();                   // Reset View
  gluPerspective(45.0, Width/Height, 1.0, 100.0);  // Do the perspective calculations. Last value = max clipping depth

  glMatrixMode(GL_MODELVIEW);         // Return to the modelview matrix
  glLoadIdentity();                   // Reset View
end;

{------------------------------------------------------------------}
{  Processes all the mouse move                                  }
{------------------------------------------------------------------}
procedure ProcessMouseMove();
begin
  if MouseMove then
  begin
    // Add Code here to execute on mouse move
    MouseMove := False;
  end;
end;

{------------------------------------------------------------------}
{  Processes all the mouse clicks                                  }
{------------------------------------------------------------------}
procedure ProcessMouseClick;
begin
  case MouseButton of
  1: // Left Mouse Button
    begin
      // Add code to execute on mouse click
      MouseButton := 0; // Cancel our mouse click (To use this procedure as a mouse down event remove this line)
    end;
  2: // Right Mouse Button
    begin
      // Add code to execute on mouse click
      MouseButton := 0; // Cancel our mouse click (To use this procedure as a mouse down event remove this line)
    end;
  3: // Middle Mouse Button
    begin
      // Add code to execute on mouse click
      MouseButton := 0;  // Cancel our mouse click (To use this procedure as a mouse down event remove this line)
    end;
  end;
end;

{------------------------------------------------------------------}
{  Processes all the keystrokes                                    }
{------------------------------------------------------------------}
procedure ProcessKeys;
begin
  if (keys[VK_UP])    then xspeed := xspeed - DeltaTime * 0.5;
  if (keys[VK_DOWN])  then xspeed := xspeed + DeltaTime * 0.5;
  if (keys[VK_RIGHT]) then yspeed := yspeed + DeltaTime * 0.5;
  if (keys[VK_LEFT])  then yspeed := yspeed - DeltaTime * 0.5;
  if (keys[VK_F1]) then BumpActive := True;
  if (keys[VK_F2]) then BumpActive := False;
  if (keys[Ord('D')]) then LightOrigin[0] := LightOrigin[0] + DeltaTime * 5;
  if (keys[Ord('A')]) then LightOrigin[0] := LightOrigin[0] - DeltaTime * 5;
  if (keys[Ord('W')]) then LightOrigin[1] := LightOrigin[1] + DeltaTime * 5;
  if (keys[Ord('S')]) then LightOrigin[1] := LightOrigin[1] - DeltaTime * 5;
  if (keys[Ord('Q')]) then LightOrigin[2] := LightOrigin[2] + DeltaTime * 5;
  if (keys[Ord('E')]) then LightOrigin[2] := LightOrigin[2] - DeltaTime * 5;
end;


{------------------------------------------------------------------}
{  Determines the applications response to the messages received  }
{------------------------------------------------------------------}
function WndProc(hWnd: HWND; Msg: UINT;  wParam: WPARAM;  lParam: LPARAM): LRESULT; stdcall;
begin
  case (Msg) of
    WM_CREATE:
      begin
        // Insert stuff you want executed when the program starts
      end;
    WM_CLOSE:
      begin
        PostQuitMessage(0);
        Result := 0
      end;
    WM_KEYDOWN:       // Set the pressed key (wparam) to equal true so we can check if its pressed
      begin
        keys[wParam] := True;
        Result := 0;
      end;
    WM_KEYUP:         // Set the released key (wparam) to equal false so we can check if its pressed
      begin
        keys[wParam] := False;
        Result := 0;
      end;
    WM_MOUSEMOVE:
      begin
        MouseMove := True;
        MoveXcoord := LOWORD(lParam);
        MoveYcoord := HIWORD(lParam);
        Result := 0;
      end;
    WM_LBUTTONDOWN:
      begin
        ReleaseCapture();   // need them here, because if mouse moves off
        SetCapture(h_Wnd);  // window and returns, it needs to reset status
        MouseButton := 1;
        Xcoord := LOWORD(lParam);
        Ycoord := HIWORD(lParam);
        Result := 0;
      end;
    WM_RBUTTONDOWN:
      begin
        ReleaseCapture();   // need them here, because if mouse moves off
        SetCapture(h_Wnd);  // window and returns, it needs to reset status
        MouseButton := 2;
        Xcoord := LOWORD(lParam);
        Ycoord := HIWORD(lParam);
        Result := 0;
      end;
    WM_MBUTTONDOWN:
      begin
        ReleaseCapture();   // need them here, because if mouse moves off
        SetCapture(h_Wnd);  // window and returns, it needs to reset status
        MouseButton := 3;
        Xcoord := LOWORD(lParam);
        Ycoord := HIWORD(lParam);
        Result := 0;
      end;
    WM_LBUTTONUP,WM_RBUTTONUP,WM_MBUTTONUP:
      begin
        ReleaseCapture();   // above
        MouseButton := 0;
        XCoord := 0;
        YCoord := 0;
        Result := 0;
      end;
    WM_SIZE:          // Resize the window with the new width and height
      begin
        if OpenGLInitialized then
         glResizeWnd(LOWORD(lParam),HIWORD(lParam));
        Result := 0;
      end;
    WM_TIMER :                     // Add code here for all timers to be used.
      begin
        if wParam = FPS_TIMER then
        begin
          FPSCount :=Round(FPSCount * 1000/FPS_INTERVAL);   // calculate to get per Second incase intercal is less or greater than 1 second
          SetWindowText(h_Wnd, PChar(WND_TITLE + '   [' + intToStr(FPSCount) + ' FPS]'));
          FPSCount := 0;
          Result := 0;
        end;
      end;
    else
      begin
      
      Result := DefWindowProc(hWnd, Msg, wParam, lParam);    // Default result if nothing happens
      end;
  end;
end;


{---------------------------------------------------------------------}
{  Properly destroys the window created at startup (no memory leaks)  }
{---------------------------------------------------------------------}
procedure glKillWnd(Fullscreen : Boolean);
begin
// vertex program srcden sil
  if VP_DOT3_Handle > 0 then
    glDeleteProgramsARB(1, @VP_DOT3_Handle);

  if Fullscreen then             // Change back to non fullscreen
  begin
    ChangeDisplaySettings(devmode(nil^), 0);
    ShowCursor(True);
  end;

  // Makes current rendering context not current, and releases the device
  // context that is used by the rendering context.
  if (not wglMakeCurrent(h_DC, 0)) then
    MessageBox(0, 'Release of DC and RC failed!', 'Error', MB_OK or MB_ICONERROR);

  // Attempts to delete the rendering context
  if (not wglDeleteContext(h_RC)) then
  begin
    MessageBox(0, 'Release of rendering context failed!', 'Error', MB_OK or MB_ICONERROR);
    h_RC := 0;
  end;

  // Attemps to release the device context
  if ((h_DC > 0) and (ReleaseDC(h_Wnd, h_DC) = 0)) then
  begin
    MessageBox(0, 'Release of device context failed!', 'Error', MB_OK or MB_ICONERROR);
    h_DC := 0;
  end;

  // Attempts to destroy the window
  if ((h_Wnd <> 0) and (not DestroyWindow(h_Wnd))) then
  begin
    MessageBox(0, 'Unable to destroy window!', 'Error', MB_OK or MB_ICONERROR);
    h_Wnd := 0;
  end;

  // Attempts to unregister the window class
  if (not UnRegisterClass('OpenGL', hInstance)) then
  begin
    MessageBox(0, 'Unable to unregister window class!', 'Error', MB_OK or MB_ICONERROR);
    hInstance := 0;
  end;
end;


{--------------------------------------------------------------------}
{  Creates the window and attaches a OpenGL rendering context to it  }
{--------------------------------------------------------------------}
function glCreateWnd(Width, Height : Integer; Fullscreen : Boolean; PixelDepth : Integer) : Boolean;
var
  wndClass : TWndClass;         // Window class
  dwStyle : DWORD;              // Window styles
  dwExStyle : DWORD;            // Extended window styles
  dmScreenSettings : DEVMODE;   // Screen settings (fullscreen, etc...)
  PixelFormat : GLuint;         // Settings for the OpenGL rendering
  h_Instance : HINST;           // Current instance
  pfd : TPIXELFORMATDESCRIPTOR;  // Settings for the OpenGL window
begin
  InitOpenGL;                               // New call to initialize and bind the OpenGL dll

  h_Instance := GetModuleHandle(nil);       //Grab An Instance For Our Window
  ZeroMemory(@wndClass, SizeOf(wndClass));  // Clear the window class structure

  with wndClass do                    // Set up the window class
  begin
    style         := CS_HREDRAW or    // Redraws entire window if length changes
                     CS_VREDRAW or    // Redraws entire window if height changes
                     CS_OWNDC;        // Unique device context for the window
    lpfnWndProc   := @WndProc;        // Set the window procedure to our func WndProc
    hInstance     := h_Instance;
    hCursor       := LoadCursor(0, IDC_ARROW);
    lpszClassName := 'OpenGL';
  end;

  if (RegisterClass(wndClass) = 0) then  // Attemp to register the window class
  begin
    MessageBox(0, 'Failed to register the window class!', 'Error', MB_OK or MB_ICONERROR);
    Result := False;
    Exit
  end;

  // Change to fullscreen if so desired
  if Fullscreen then
  begin
    ZeroMemory(@dmScreenSettings, SizeOf(dmScreenSettings));
    with dmScreenSettings do begin              // Set parameters for the screen setting
      dmSize       := SizeOf(dmScreenSettings);
      dmPelsWidth  := Width;                    // Window width
      dmPelsHeight := Height;                   // Window height
      dmBitsPerPel := PixelDepth;               // Window color depth
      dmFields     := DM_PELSWIDTH or DM_PELSHEIGHT or DM_BITSPERPEL;
    end;

    // Try to change screen mode to fullscreen
    if (ChangeDisplaySettings(dmScreenSettings, CDS_FULLSCREEN) = DISP_CHANGE_FAILED) then
    begin
      MessageBox(0, 'Unable to switch to fullscreen!', 'Error', MB_OK or MB_ICONERROR);
      Fullscreen := False;
    end;
  end;

  // If we are still in fullscreen then
  if (Fullscreen) then
  begin
    dwStyle := WS_POPUP or                // Creates a popup window
               WS_CLIPCHILDREN            // Doesn't draw within child windows
               or WS_CLIPSIBLINGS;        // Doesn't draw within sibling windows
    dwExStyle := WS_EX_APPWINDOW;         // Top level window
    ShowCursor(False);                    // Turn of the cursor (gets in the way)
  end
  else
  begin
    dwStyle := WS_OVERLAPPEDWINDOW or     // Creates an overlapping window
               WS_CLIPCHILDREN or         // Doesn't draw within child windows
               WS_CLIPSIBLINGS;           // Doesn't draw within sibling windows
    dwExStyle := WS_EX_APPWINDOW or       // Top level window
                 WS_EX_WINDOWEDGE;        // Border with a raised edge
  end;

  // Attempt to create the actual window
  h_Wnd := CreateWindowEx(dwExStyle,      // Extended window styles
                          'OpenGL',       // Class name
                          WND_TITLE,      // Window title (caption)
                          dwStyle,        // Window styles
                          0, 0,           // Window position
                          Width, Height,  // Size of window
                          0,              // No parent window
                          0,              // No menu
                          h_Instance,     // Instance
                          nil);           // Pass nothing to WM_CREATE

  if h_Wnd = 0 then
  begin
    glKillWnd(Fullscreen);                // Undo all the settings we've changed
    MessageBox(0, 'Unable to create window!', 'Error', MB_OK or MB_ICONERROR);
    Result := False;
    Exit;
  end;

  // Try to get a device context
  h_DC := GetDC(h_Wnd);
  if (h_DC = 0) then
  begin
    glKillWnd(Fullscreen);
    MessageBox(0, 'Unable to get a device context!', 'Error', MB_OK or MB_ICONERROR);
    Result := False;
    Exit;
  end;

  // Settings for the OpenGL window
  with pfd do
  begin
    nSize           := SizeOf(TPIXELFORMATDESCRIPTOR); // Size Of This Pixel Format Descriptor
    nVersion        := 1;                    // The version of this data structure
    dwFlags         := PFD_DRAW_TO_WINDOW    // Buffer supports drawing to window
                       or PFD_SUPPORT_OPENGL // Buffer supports OpenGL drawing
                       or PFD_DOUBLEBUFFER;  // Supports double buffering
    iPixelType      := PFD_TYPE_RGBA;        // RGBA color format
    cColorBits      := PixelDepth;           // OpenGL color depth
    cRedBits        := 0;                    // Number of red bitplanes
    cRedShift       := 0;                    // Shift count for red bitplanes
    cGreenBits      := 0;                    // Number of green bitplanes
    cGreenShift     := 0;                    // Shift count for green bitplanes
    cBlueBits       := 0;                    // Number of blue bitplanes
    cBlueShift      := 0;                    // Shift count for blue bitplanes
    cAlphaBits      := 0;                    // Not supported
    cAlphaShift     := 0;                    // Not supported
    cAccumBits      := 0;                    // No accumulation buffer
    cAccumRedBits   := 0;                    // Number of red bits in a-buffer
    cAccumGreenBits := 0;                    // Number of green bits in a-buffer
    cAccumBlueBits  := 0;                    // Number of blue bits in a-buffer
    cAccumAlphaBits := 0;                    // Number of alpha bits in a-buffer
    cDepthBits      := 16;                   // Specifies the depth of the depth buffer
    cStencilBits    := 0;                    // Turn off stencil buffer
    cAuxBuffers     := 0;                    // Not supported
    iLayerType      := PFD_MAIN_PLANE;       // Ignored
    bReserved       := 0;                    // Number of overlay and underlay planes
    dwLayerMask     := 0;                    // Ignored
    dwVisibleMask   := 0;                    // Transparent color of underlay plane
    dwDamageMask    := 0;                     // Ignored
  end;

  // Attempts to find the pixel format supported by a device context that is the best match to a given pixel format specification.
  PixelFormat := ChoosePixelFormat(h_DC, @pfd);
  if (PixelFormat = 0) then
  begin
    glKillWnd(Fullscreen);
    MessageBox(0, 'Unable to find a suitable pixel format', 'Error', MB_OK or MB_ICONERROR);
    Result := False;
    Exit;
  end;

  // Sets the specified device context's pixel format to the format specified by the PixelFormat.
  if (not SetPixelFormat(h_DC, PixelFormat, @pfd)) then
  begin
    glKillWnd(Fullscreen);
    MessageBox(0, 'Unable to set the pixel format', 'Error', MB_OK or MB_ICONERROR);
    Result := False;
    Exit;
  end;

  // Create a OpenGL rendering context
  h_RC := wglCreateContext(h_DC);
  if (h_RC = 0) then
  begin
    glKillWnd(Fullscreen);
    MessageBox(0, 'Unable to create an OpenGL rendering context', 'Error', MB_OK or MB_ICONERROR);
    Result := False;
    Exit;
  end;

  // Makes the specified OpenGL rendering context the calling thread's current rendering context
  if (not wglMakeCurrent(h_DC, h_RC)) then
  begin
    glKillWnd(Fullscreen);
    MessageBox(0, 'Unable to activate OpenGL rendering context', 'Error', MB_OK or MB_ICONERROR);
    Result := False;
    Exit;
  end;
  // Read And Assign Extentions
  ReadExtensions;
  ReadImplementationProperties;
  OpenGLInitialized := True;
  // Initializes the timer used to calculate the FPS
  SetTimer(h_Wnd, FPS_TIMER, FPS_INTERVAL, nil);

  // Settings to ensure that the window is the topmost window
  ShowWindow(h_Wnd, SW_SHOW);
  SetForegroundWindow(h_Wnd);
  SetFocus(h_Wnd);

  // Ensure the OpenGL window is resized properly
  glResizeWnd(Width, Height);
  glInit();

  Result := True;
end;


{--------------------------------------------------------------------}
{  Main message loop for the application                             }
{--------------------------------------------------------------------}
function WinMain(hInstance : HINST; hPrevInstance : HINST;
                 lpCmdLine : PChar; nCmdShow : Integer) : Integer; stdcall;
var
  msg : TMsg;
  finished : Boolean;
  DemoStart, LastTime : DWord;
  tamekran: Boolean;
begin
  finished := False;

  tamekran := (MessageBox(h_Wnd, 'Tam ekran modunda altrmak ister misiniz?',
                          nil, MB_YESNO or MB_ICONQUESTION) = idYes);

  // Perform application initialization:
  if not glCreateWnd(800, 600, tamekran, 32) then
  begin
    Result := 0;
    Exit;
  end;

  DemoStart := GetTickCount();            // Get Time when demo started

  // Main message loop:
  while not finished do
  begin
    if (PeekMessage(msg, 0, 0, 0, PM_REMOVE)) then // Check if there is a message for this window
    begin
      if (msg.message = WM_QUIT) then     // If WM_QUIT message received then we are done
        finished := True
      else
      begin                               // Else translate and dispatch the message to this window
  	TranslateMessage(msg);
        DispatchMessage(msg);
      end;
    end
    else
    begin
      Inc(FPSCount);                      // Increment FPS Counter

      DeltaTime := ElapsedTime;
      LastTime :=ElapsedTime;
      ElapsedTime :=GetTickCount() - DemoStart;     // Calculate Elapsed Time
      ElapsedTime :=(LastTime + ElapsedTime) DIV 2; // Average it out for smoother movement
      DeltaTime := (ElapsedTime - DeltaTime) * 0.001;

      glDraw();                           // Draw the scene
      SwapBuffers(h_DC);                  // Display the scene

      if (keys[VK_ESCAPE]) then           // If user pressed ESC then set finised TRUE
        finished := True
      else
      begin
        ProcessKeys;                      // Check for any other key Pressed
        ProcessMouseMove;                 // Check for mouse move
        ProcessMouseClick;                // Check for Mouse Click
      end;
    end;
  end;
  glKillWnd(FALSE);
  Result := msg.wParam;
end;


begin
  WinMain( hInstance, hPrevInst, CmdLine, CmdShow );
end.
