/*! 
	\file	TextureManager.h
	\brief	Textures are stored and managed here.
*/

#ifndef TEXTUREMANAGER_H_
#define TEXTUREMANAGER_H_

#include <map>
#include <string>
#include "ITexture.h"
#include "ITextureLoader.h"

/*!
	\brief Textures are stored and managed here.
	\ingroup XSceneGraph
	\author Koray Balci
	\version 1.0
	\date   May 2003

	Textures (ITexture instances) are loaded, stored and managed through here. 
	All the textures are stored in a hash map (currently) with a string representing 
	the name as the key of the hash. TextureManager is a singleton class.

	Here is a sample usage for loading a texture;
	\code
	TextureManager::getInstance()->load("..\\john_lo_obj02.bmp", "john_lo_obj02");
	// this is optional but good to do, associate the texture to a Drawable
	Drawable* pDrawable = new Drawable;
	pDrawable->setTexName("john_lo_obj00");
	\endcode

	And here is how we access the loaded texture (from RendererGL);
	\code
	const ITexture* pTexture = TextureManager::getInstance()->getTexture(pDrawable->getTexName());
	\endcode
*/
namespace XSceneGraph{

class TextureManager
{
	typedef std::map<std::string, const ITexture*> TEXSTORAGE;
	TEXSTORAGE m_Storage;
	static TextureManager* m_pThis;
	XStream::ITextureLoader* m_pTexLoader;
	
	TextureManager(void);
	~TextureManager(void);
public:
	const ITexture* load(const std::string& filename, const std::string& texname );
	void unLoad(const ITexture* pTexture);
	void unLoad(const std::string& name);

	static void releaseInstance();
	static TextureManager* getInstance();
	void destroyAll();
	void registerTexture(const ITexture* pTexture);
	void unregisterTexture(const ITexture* pTexture);
	const ITexture* getTexture(const std::string& name) const;
};
}
#endif //TEXTUREMANAGER_H_